<?php

namespace Mnv\Admin\Controllers;

use Mnv\Core\Auth\Exceptions\EmailNotVerifiedException;
use Mnv\Core\Auth\Exceptions\UnknownIdException;
use Symfony\Component\HttpFoundation\Response;
use Mnv\Core\Security\Csrf\Exceptions\InvalidCsrfTokenException;

use Mnv\Core\Auth\AdminAuthorize;
use Mnv\Core\Auth\Errors\UsernameRequiredError;
use Mnv\Core\Auth\Exceptions\UsernameBlockedException;
use Mnv\Core\Auth\Exceptions\InvalidPasswordException;
use Mnv\Core\Auth\Exceptions\TooManyRequestsException;
use Mnv\Core\Auth\Exceptions\UnknownUsernameException;

/**
 * Class AuthAdmin
 * @package Mnv\Admin\Controllers
 */
class AuthAdmin extends AdminAuthorize
{

    /** Constructor initializes admin authentication and design, and handles request. */
    public function __construct()
    {

        parent::__construct();

        // initializes authentication and loads language settings.
        $this->initializeAuth();

        // Retrieves the remember duration for the session.
        $this->getRememberDuration();

        // Handles login or logout request based on the action.
        $this->handleRequest();

        //  Assigns data to the Smarty template engine.
        $this->assignSmartyVariables();
    }


    /** Handles login or logout request based on the action. */
    private function handleRequest(): void
    {
        $action = request()->get('action', '');
        switch ($action) {
            case 'login':
                $this->processAdminLogin();
                break;
            case 'logout':
                $this->processLogout();
                break;
        }
    }

    /** Processes admin login, sanitizes data, and handles exceptions. */
    private function processAdminLogin(): void
    {
        try {
            $formToken = request()->get('token', '');
            $this->securityCsrfToken->check('token', $formToken, 60 , true);

            try {
                $this->sanitizeLoginData();
                $this->auth->loginWithUsername($this->login['login'], $this->login['password'], $this->rememberDuration);
                $this->redirectToAdminPage();
            } catch (UsernameRequiredError $e) {
                $this->addLoginError('enter_username');  // пустой логин
            } catch (UnknownUsernameException|UnknownIdException $e) {
                $this->addLoginError('user_not_found'); // Пользователь не найден.
            } catch (InvalidPasswordException $e) {
                $this->addLoginError('incorrect_credentials'); // Пользователь найден, но не верный пароль.
            } catch (EmailNotVerifiedException $e) {
                $this->addLoginError('email_not_verified');  // email не подтвержден
            } catch (TooManyRequestsException $e) {
                $this->addLoginError('too_many_requests');    // слишком много запросов
            } catch (UsernameBlockedException $e) {
                $this->errors[] = $e->getMessage(); // блокировка пользователя
            }
        } catch (InvalidCsrfTokenException $e ) {
            $this->response['status'] = Response::HTTP_FORBIDDEN;
            $this->errors[] = $e->getMessage();
        }
    }


    /** Fetches the login page or redirects if the admin is already logged in. */
    public function fetch()
    {

        /** generate url for type content */
        $this->smarty->assign('adminUrl', $this->adminUrl);

        if ($this->auth->isLoggedIn()) {
            $this->redirectToAdminPage();
        }

        try {
            return $this->smarty->fetch('views/auth/index.tpl');
        } catch (\SmartyException | \Exception $e) {
            return $e->getMessage();
        }
    }

}










