<?php

namespace Mnv\Admin\Controllers;

use Mnv\Core\Auth\Exceptions\EmailNotVerifiedException;
use Mnv\Core\Auth\Exceptions\RefererExceptions;
use Mnv\Core\Auth\Exceptions\UnknownIdException;
use Mnv\Models\UserGroups;

use Symfony\Component\HttpFoundation\Response;
use Mnv\Core\Security\Csrf\Exceptions\InvalidCsrfTokenException;

use Mnv\Core\Auth\AdminAuthorize;
use Mnv\Core\Auth\Errors\UsernameRequiredError;
use Mnv\Core\Auth\Exceptions\InvalidPasswordException;
use Mnv\Core\Auth\Exceptions\UsernameBlockedException;
use Mnv\Core\Auth\Exceptions\TooManyRequestsException;
use Mnv\Core\Auth\Exceptions\UnknownUsernameException;

/**
 * Class AuthAjaxAdmin
 * @package Mnv\Admin\Controllers
 */

class AuthAjaxAdmin extends AdminAuthorize
{
    /** Constructor initializes admin authentication and design, and handles request. */
    public function __construct()
    {
        parent::__construct();

        // Initializes authentication and loads language settings.
        $this->initializeAuth();

        if (!request()->ajax()) {
            $this->sendErrorResponse(Response::HTTP_FORBIDDEN, 'Invalid request.');
        }



        try {
            $formToken = $_SERVER['HTTP_X_CSRF_TOKEN'] ?? request()->get('token', null);

            // Validates the CSRF token.
            $this->validateCsrfToken($formToken);

            // Validates the Referer header.
            $this->validateReferer();

            // Retrieves the remember duration for the session.
            $this->getRememberDuration();

            // Processes admin login.
            $this->processAdminLoginAjax();
        }
        catch (InvalidCsrfTokenException $e) {
            $this->sendErrorResponse(Response::HTTP_FORBIDDEN, $e->getMessage() . ' Form ignored.');
        } catch (RefererExceptions $e) { // Exception Referer header.
            $this->sendErrorResponse(Response::HTTP_FORBIDDEN, $e->getMessage());
        }
    }

    /**
     * Validates the CSRF token.
     *
     * @throws InvalidCsrfTokenException
     */
    private function validateCsrfToken($token): void
    {
        $this->securityCsrfToken->check('token', $token, null, true);
    }

    /**
     * Validates the Referer header.
     *
     * @throws RefererExceptions
     */
    private function validateReferer(): void
    {
        if (stripos($_SERVER['HTTP_REFERER'], $_SERVER['SERVER_NAME']) === false) {
            throw new RefererExceptions('Invalid Referer.');
        }
    }

    /**
     * Processes admin login, sanitizes data, and handles exceptions.
     */
    private function processAdminLoginAjax(): void
    {
        try {
            $this->sanitizeLoginData();
            if (isset($this->login['login'])) {
                $this->auth->loginWithUsername($this->login['login'], $this->login['password'], $this->rememberDuration);
            }
            else if (isset($this->login['phone'])) {
                $this->auth->loginWithPhone($this->login['phone'], $this->login['password'], $this->rememberDuration);
            }

            if ($this->auth->hasAnyRole(array_keys(UserGroups::arrayGroup()))) {
                 $this->sendResponse([
                    'status' => Response::HTTP_OK,
                    'message' => lang('login:messages:welcome'),
                    'redirect' => '/admin/main'
                ]);
            }

        } catch (UsernameRequiredError $e) {
            $this->handleLoginError('enter_username');  // пустой логин
        } catch (UnknownUsernameException|UnknownIdException $e) {
            $this->handleLoginError('user_not_found');  // Пользователь не найден.
        } catch (InvalidPasswordException $e) {
            $this->handleLoginError('incorrect_credentials'); // Пользователь найден, но не верный пароль.
        }  catch (EmailNotVerifiedException $e) {
            $this->handleLoginError('email_not_verified');  // email не подтверждена
        } catch (TooManyRequestsException $e) {
            $this->handleLoginError('too_many_requests');  // слишком много запросов
        } catch (UsernameBlockedException $e) {
            $this->handleBlockedUser($e); // блокировка пользователя
        }
    }

    /**
     * Handles login error by adding the corresponding error message.
     */
    private function handleLoginError(string $errorKey): void
    {
        $this->addLoginErrorAjax($errorKey);
    }

    /**
     * Handles user blocking during login.
     */
    private function handleBlockedUser(UsernameBlockedException $e): void
    {
        if ($this->auth->blockHacking) {
        $attempts = $this->auth->limit();
        $remainingAttempts = $this->auth->maxAttempts - $attempts;

        $this->sendResponse([
            'status' => Response::HTTP_PRECONDITION_FAILED,
            'message' => $e->getMessage(),
            'attempts' => $attempts,
            'blocked' =>  $this->auth->blockHacking && $attempts >= $this->auth->maxAttempts,
            'limit' => ($remainingAttempts > 0 && $remainingAttempts < 6)
                ? pluralLanguage($remainingAttempts, 'remaining') . " $remainingAttempts " . pluralLanguage($remainingAttempts, 'attempt')
                : null
        ]);
        } else {
            $this->sendResponse([
                'status' => Response::HTTP_PRECONDITION_FAILED,
                'message' => $e->getMessage(),
                'blocked' => false,
            ]);
        }
    }

    /**
     * Adds error messages for login attempts.
     */
    private function addLoginErrorAjax(string $errorKey): void
    {
        if ($this->auth->blockHacking) {
            $attempts = $this->auth->limit();
            $remainingAttempts = $this->auth->maxAttempts - $attempts;

            $response = [
                'status' => Response::HTTP_PRECONDITION_FAILED,
                'message' => lang("login:errors:$errorKey"),
                'attempts' => $attempts,
                'blocked' => $attempts >= $this->auth->maxAttempts,
                'limit' => ($remainingAttempts > 0 && $remainingAttempts < 6)
                    ? pluralLanguage($remainingAttempts, 'remaining')." $remainingAttempts ".pluralLanguage($remainingAttempts, 'attempt')
                    : null
            ];

            if ($attempts == $this->auth->maxAttempts - 1) {
                $response['message'] = lang('login:errors:today_attempts');
            }
        } else {
            $response = [
                'status' => Response::HTTP_PRECONDITION_FAILED,
                'message' => lang("login:errors:$errorKey"),
                'blocked' => false,
            ];
        }

        $this->sendResponse($response);
    }

    /**
     * Sends a JSON response.
     */
    private function sendResponse(array $response): void
    {
        response()->json($response, Response::HTTP_OK, [
            'Content-Type' => 'application/json;charset=UTF-8',
            'Charset' => 'utf-8'
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT)->send();
        exit();
    }

    /**
     * Sends an error response with a given message and status.
     *
     * @param  int  $status
     * @param  string  $message
     * @return void
     */
    private function sendErrorResponse(int $status, string $message): void
    {
        $this->sendResponse([
            'status' => $status,
            'message' => $message
        ]);
    }
}





