<?php

namespace Mnv\Core\Auth;

use Mnv\Core\Auth\Exceptions\ValidatePasswordException;
use Mnv\Core\UserInfo;
use Mnv\Core\Utilities\Base64\Base64;
use Mnv\Core\Utilities\Cookie\Cookie;
use Mnv\Core\Utilities\Cookie\Session;

use Mnv\Core\Auth\Exceptions\UnknownIdException;
use Mnv\Core\Auth\Exceptions\NotLoggedInException;
use Mnv\Core\Auth\Exceptions\InvalidPasswordException;
use Mnv\Core\Auth\Exceptions\TooManyRequestsException;
use Mnv\Core\Auth\Exceptions\UnknownUsernameException;
use Mnv\Core\Auth\Exceptions\UsernameBlockedException;
use Mnv\Core\Auth\Exceptions\EmailNotVerifiedException;

use Mnv\Core\Auth\Errors\AuthError;
use Mnv\Core\Auth\Errors\UsernameRequiredError;
use Mnv\Core\Auth\Errors\HeadersAlreadySentError;

/**
 * Class Administration
 * @package Mnv\Core\Auth
 */
class Administration extends AdminManager
{

    /** @var mixed|string|null текущий IP-адрес пользователя */
    private $ipAddress;
    /** @var bool должно ли регулирование быть включено (например, в производстве) или отключено (например, во время разработки) */
    private bool $throttling;
    /** @var int интервал в секундах, по истечении которого необходимо повторно синхронизировать данные сеанса с его официальным источником в базе данных */
    private int $sessionResyncInterval;
    /** @var string имя файла cookie, используемого для функции "запомнить меня" */
    private string $rememberCookieName;

    /**
     * Administration constructor.
     *
     * @param string|null $ipAddress  (optional) IP-адрес, который следует использовать вместо параметра по умолчанию (если таковой имеется), например, при использовании прокси-сервера
     * @param bool|null $throttling (optional) следует ли включать регулирование (например, в процессе производства) или отключать (например, во время разработки)
     * @param int|null $sessionResyncInterval (optional) интервал в секундах, по истечении которого необходимо повторно синхронизировать данные сеанса с их авторитетным источником в базе данных
     */
    public function __construct(string $ipAddress = null, bool $throttling = null, int $sessionResyncInterval = null)
    {
        parent::__construct();

        $this->ipAddress = !empty($ipAddress) ? $ipAddress : ($_SERVER['REMOTE_ADDR'] ?? null);
        $this->throttling = !isset($throttling) || $throttling;
        $this->sessionResyncInterval = !empty($sessionResyncInterval) ? $sessionResyncInterval : (60 * 5);
        $this->rememberCookieName = self::createRememberCookieName();

        $this->initSessionIfNecessary();
        $this->enhanceHttpSecurity();

        $this->processRememberDirective();
        $this->resyncSessionIfNecessary();

    }

    /**
     * Инициализирует сеанс и устанавливает правильную конфигурацию
     */
    private function initSessionIfNecessary()
    {
        if (session_status() === \PHP_SESSION_NONE) {
            // использовать файлы cookie для хранения идентификаторов сеансов
            ini_set('session.use_cookies', 1);
            // использовать только файлы cookie (не отправлять идентификаторы сеанса в URL-адресах)
            ini_set('session.use_only_cookies', 1);
            // не отправлять идентификаторы сеанса в URL-адресах
            ini_set('session.use_trans_sid', 0);

            // запустить сеанс (запрашивает запись cookie на клиенте)
            @Session::start();
        }
    }

    /**
     * Повышает безопасность приложения по протоколу HTTP(S), задавая определенные заголовки.
     */
    private function enhanceHttpSecurity()
    {
        // удалить раскрытие версии PHP (по крайней мере, где это возможно)
        header_remove('X-Powered-By');

        // если пользователь вошел в систему
        if ($this->isLoggedIn()) {
            // предотвратить кликджекинг
            header('X-Frame-Options: SAMEORIGIN');
            // предотвращение прослушивания контента (сниффинг MIME)
            header('X-Content-Type-Options: nosniff');
            // защита от XSS-атак
            header('X-XSS-Protection: 1; mode=block');
            // использовать политику безопасности контента (CSP) для защиты от XSS и других атак
//          header("Content-Security-Policy: default-src 'self'; script-src 'self'");
            // предотвратить кэширование потенциально конфиденциальных данных
            header('Cache-Control: no-store, no-cache, must-revalidate', true);
            header('Expires: Thu, 19 Nov 1981 00:00:00 GMT', true);
            header('Pragma: no-cache', true);

            // включить Strict-Transport-Security для принудительного использования HTTPS
            if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') {
                header('Strict-Transport-Security: max-age=31536000; includeSubDomains');
            }
        }
    }

    /** Проверяет наличие установленной директивы "запомнить меня" и обрабатывает автоматический вход (при необходимости) */
    private function processRememberDirective()
    {
        // если пользователь еще не вошел в систему

        if (!$this->isLoggedIn()) {

            // если в настоящее время нет файла cookie для функции «запомнить меня»
            if (!isset($_COOKIE[$this->rememberCookieName])) {
                // если был обнаружен старый файл cookie для этой функции от версий v1.x.x до v6.x.x
                if (isset($_COOKIE['auth_remember'])) {
                    // вместо этого используйте значение из этого старого файла cookie
                    $_COOKIE[$this->rememberCookieName] = $_COOKIE['auth_remember'];
                }
            }

            // если запоминающийся cookie установлен
            if (isset($_COOKIE[$this->rememberCookieName])) {
                // предполагать, что файл cookie и его содержимое недействительны, пока не будет доказано обратное
                $valid = false;

                // разделить содержимое файла cookie на селектор и токен
                $parts = \explode(self::COOKIE_CONTENT_SEPARATOR, $_COOKIE[$this->rememberCookieName], 2);

                // если были найдены и селектор, и токен
                if (!empty($parts[0]) && !empty($parts[1])) {
                    $rememberData = connect('users_remembered')
                        ->join('users','user', '=', 'userId')
                        ->select('user, token, expires, loginName, phone, status, groupId, forceLogout')
                        ->where('selector','=', $parts[0])
                        ->get('array');


                    if (!empty($rememberData)) {
                        if ($rememberData['expires'] >= \time()) {
                            if (\password_verify($parts[1], $rememberData['token'])) {
                                // cookie и его содержимое теперь доказали свою действительность
                                $valid = true;

                                $this->onLoginSuccessful($rememberData['user'], $rememberData['loginName'], $rememberData['status'], $rememberData['groupId'], $rememberData['forceLogout'], true);
                            }
                        }
                    }
                }

                // если файл cookie или его содержимое были недействительными
                if (!$valid) {
                    // пометьте файл cookie, чтобы предотвратить дальнейшие попытки
                    $this->setRememberCookie('', '', \time() + 60 * 60 * 24 * 365.25);
                }
            }
        }
    }

    private function resyncSessionIfNecessary()
    {
        // если пользователь вошел в систему
        if ($this->isLoggedIn()) {
            // следующее поле сеанса, возможно, не было инициализировано для сеансов, которые уже существовали до введения этой функции
            if (!isset($_SESSION[self::SESSION_FIELD_LAST_RESYNC])) {
                $_SESSION[self::SESSION_FIELD_LAST_RESYNC] = 0;
            }

            // если пришло время для повторной синхронизации
            if (($_SESSION[self::SESSION_FIELD_LAST_RESYNC] + $this->sessionResyncInterval) <= \time()) {
                // снова получить достоверные данные из базы данных
                $authoritativeData = connect('users')->select('email, loginName, status, groupId, forceLogout')->where('userId', $this->getUserId())->get('array');

                // если данные пользователя были найдены
                if (!empty($authoritativeData)) {
                    // следующее поле сеанса, возможно, не было инициализировано для сеансов, которые уже существовали до введения этой функции
                    if (!isset($_SESSION[self::SESSION_FIELD_FORCE_LOGOUT])) {
                        $_SESSION[self::SESSION_FIELD_FORCE_LOGOUT] = 0;
                    }

                    // если счетчик, отслеживающий принудительные выходы из системы, был увеличен
                    if ($authoritativeData['forceLogout'] > $_SESSION[self::SESSION_FIELD_FORCE_LOGOUT]) {
                        // пользователь должен выйти из системы
                        $this->logOut();
                    }
                    // если счетчик, отслеживающий принудительные выходы из системы, остался неизменным
                    else {
                        // данные сеанса необходимо обновить
                        $_SESSION[self::SESSION_FIELD_USERNAME] = $authoritativeData['loginName'];
                        $_SESSION[self::SESSION_FIELD_STATUS]   = (int) $authoritativeData['status'];
                        $_SESSION[self::SESSION_FIELD_ROLES]    = (int) $authoritativeData['groupId'];

                        // помните, что мы только что выполнили необходимую повторную синхронизацию
                        $_SESSION[self::SESSION_FIELD_LAST_RESYNC] = \time();
                    }
                }
                // если данные для пользователя не найдены
                else {
                    // их учетная запись могла быть удалена, поэтому они должны выйти из системы
                    $this->logOut();
                }
            }
        }
    }


    /**
     * Вход в систему с помощью phone and password
     * При использовании этого метода для аутентификации пользователей следует убедиться, что имена пользователей уникальны.
     *
     * @param  string  $phone  the user's phone
     * @param  string|null  $password  the user's password
     * @param  bool  $rememberDuration  (optional) время в секундах, в течение которого пользователь остается в системе ("remember me"), e.g. `60 * 60 * 24 * 365.25` на один год
     * @param  callable|null  $onBeforeSuccess  (optional) функция, которая получает идентификатор пользователя в качестве единственного параметра и выполняется до успешной аутентификации; должна возвращать `true` для продолжения или `false` для отмены
     *
     * @throws EmailNotVerifiedException
     * @throws InvalidPasswordException if the password was invalid
     * @throws TooManyRequestsException if the number of allowed attempts/requests has been exceeded
     * @throws UnknownIdException
     * @throws UnknownUsernameException if the specified loginName does not exist
     * @throws UsernameBlockedException
     * @throws UsernameRequiredError if the specified loginName does not exist
     */
    public function loginWithPhone(string $phone, ?string $password, bool $rememberDuration = false, callable $onBeforeSuccess = null)
    {
        $this->throttle(['attemptToLogin', 'phone', $phone], 500, (60 * 60 * 24), null, true);

        // Получаем или создаем идентификатор устройства
        $this->deviceId = $this->createDeviceId();

        $this->authenticate($password, $phone, null, 'phone', $rememberDuration, $onBeforeSuccess);
    }

    /**
     * Вход в систему с помощью loginName and password
     * При использовании этого метода для аутентификации пользователей следует убедиться, что имена пользователей уникальны.
     *
     * @param  string  $loginName
     * @param  string|null  $password  the user's password
     * @param  bool  $rememberDuration  (optional) время в секундах, в течение которого пользователь остается в системе ("remember me"), e.g. `60 * 60 * 24 * 365.25` на один год
     * @param  callable|null  $onBeforeSuccess  (optional) функция, которая получает идентификатор пользователя в качестве единственного параметра и выполняется до успешной аутентификации; должна возвращать `true` для продолжения или `false` для отмены
     *
     * @throws EmailNotVerifiedException
     * @throws InvalidPasswordException if the password was invalid
     * @throws TooManyRequestsException if the number of allowed attempts/requests has been exceeded
     * @throws UnknownIdException
     * @throws UnknownUsernameException if the specified loginName does not exist
     * @throws UsernameBlockedException
     * @throws UsernameRequiredError if the specified loginName does not exist
     */
    public function loginWithUsername(string $loginName, ?string $password, bool $rememberDuration = false, callable $onBeforeSuccess = null) {

        $this->throttle([ 'attemptToLogin', 'loginName', $loginName ], 500, (60 * 60 * 24), null, true);

        // Получаем или создаем идентификатор устройства
        $this->deviceId = $this->createDeviceId();

        $this->authenticate($password, null, $loginName, 'loginName', $rememberDuration, $onBeforeSuccess);
    }

    /**
     * Вход в систему с помощью loginName and password
     * При использовании этого метода для аутентификации пользователей следует убедиться, что имена пользователей уникальны.
     *
     * @param  string  $email
     * @param  string|null  $password  the user's password
     * @param  bool  $rememberDuration  (optional) время в секундах, в течение которого пользователь остается в системе ("remember me"), e.g. `60 * 60 * 24 * 365.25` на один год
     * @param  callable|null  $onBeforeSuccess  (optional) функция, которая получает идентификатор пользователя в качестве единственного параметра и выполняется до успешной аутентификации; должна возвращать `true` для продолжения или `false` для отмены
     *
     * @throws EmailNotVerifiedException
     * @throws InvalidPasswordException if the password was invalid
     * @throws TooManyRequestsException if the number of allowed attempts/requests has been exceeded
     * @throws UnknownIdException
     * @throws UnknownUsernameException if the specified loginName does not exist
     * @throws UsernameBlockedException
     * @throws UsernameRequiredError if the specified loginName does not exist
     */
    public function loginWithEmail(string $email, ?string $password, bool $rememberDuration = false, callable $onBeforeSuccess = null) {

        $this->throttle([ 'attemptToLogin', 'email', $email ], 500, (60 * 60 * 24), null, true);

        // Получаем или создаем идентификатор устройства
        $this->deviceId = $this->createDeviceId();

        $this->authenticate($password, null, $email, 'email', $rememberDuration, $onBeforeSuccess);
    }


    /**
     * Выполняет выход пользователя из системы
     */
    public function logOut()
    {
        // если user вошел в систему
        if ($this->isLoggedIn()) {
            // получить любую локально существующую директиву remember
            $rememberDirectiveSelector = $this->getRememberDirectiveSelector();

            // если такая директива remember существует
            if (isset($rememberDirectiveSelector)) {
                // удалить директиву local remember
                $this->deleteRememberDirectiveForUserById($this->getUserId(), $rememberDirectiveSelector);
            }
            // удалить все переменные session, поддерживаемые этой библиотекой
            $this->clearAdminSessions();
        }
    }

    /**
     * Выполняет выход пользователя из всех остальных сеансов (кроме текущего)
     *
     * @throws NotLoggedInException if the user is not currently signed in
     */
    public function logOutEverywhereElse()
    {
        if (!$this->isLoggedIn()) {
            throw new NotLoggedInException();
        }

        // определить дату истечения срока действия любой локально существующей директивы запоминания
        $previousRememberDirectiveExpiry = $this->getRememberDirectiveExpiry();

        // запланировать принудительный выход из системы во всех сеансах
        $this->forceLogoutForUserById($this->getUserId());

//        $this->lockForUserById($this->getUserId());

        // следующее поле сеанса, возможно, не было инициализировано для сеансов, которые уже существовали до введения этой функции
        if (!isset($_SESSION[self::SESSION_FIELD_FORCE_LOGOUT])) {
            $_SESSION[self::SESSION_FIELD_FORCE_LOGOUT] = 0;
        }

        // убедитесь, что мы просто пропустим или проигнорируем следующий принудительный выход из системы (который мы только что вызвали) в текущем сеансе
        $_SESSION[self::SESSION_FIELD_FORCE_LOGOUT]++;

        // повторно сгенерировать идентификатор сеанса, чтобы предотвратить атаки фиксации сеанса (запрашивает запись cookie на клиенте)
        Session::regenerate(true);

        // если ранее существовала директива запоминания
        if (isset($previousRememberDirectiveExpiry)) {
            // восстановить директиву со старой датой истечения срока действия, но с новыми учетными данными
            $this->createRememberDirective($this->getUserId(), $previousRememberDirectiveExpiry - \time());
        }
    }

    /**
     * Выходит из системы во всех sessions
     *
     * @throws NotLoggedInException if the user is not currently signed in
     */
    public function logOutEverywhere()
    {
        if (!$this->isLoggedIn()) {
            throw new NotLoggedInException();
        }

        // запланировать принудительный выход из системы во всех sessions
        $this->forceLogoutForUserById($this->getUserId());

        // и немедленно примените выход из системы локально
        $this->logOut();
    }

    /**
     * Уничтожает все данные сеанса
     */
    public function destroySession()
    {
        // remove all session variables without exception
        $_SESSION = [];
        // delete the session cookie
        $this->deleteSessionCookie();
        // let PHP destroy the session
        \session_destroy();
    }

    /**
     * Удаение всех сессий
     * @return void
     */
    protected function clearAdminSessions()
    {
        if (session_status() === PHP_SESSION_ACTIVE) {
            foreach ($_SESSION as $key => $value) {
                if (strpos($key, 'admin.') === 0) {
                    unset($_SESSION[$key]);
                }
            }
        }
    }


    /**
     * Создает новую директиву, удерживающую пользователя в системе ("remember me")
     *
     * @param  int  $userId
     * @param  int  $duration
     *
     * @return void
     */
    private function createRememberDirective(int $userId, int $duration)
    {
        $selector       = self::createRandomString(24);
        $token          = self::createRandomString(32);
        $tokenHashed    = \password_hash($token, \PASSWORD_DEFAULT);
        $expires        = \time() + $duration;

        connect('users_remembered')->insert([
            'user'      => $userId,
            'selector'  => $selector,
            'token'     => $tokenHashed,
            'expires'   => $expires
        ]);

        $this->setRememberCookie($selector, $token, $expires);
    }

    /**
     * @param int $userId
     * @param string|null $selector
     * @throws AuthError
     */
    protected function deleteRememberDirectiveForUserById(int $userId, ?string $selector = null)
    {
        parent::deleteRememberDirectiveForUserById($userId, $selector);

        $this->setRememberCookie(null, null, \time() - 3600);
    }

    /**
     * Устанавливает или обновляет файл cookie, который управляет token "remember me" token
     *
     * @param string|null $selector the selector from the selector/token pair
     * @param string|null $token the token from the selector/token pair
     * @param int $expires the UNIX time in seconds which the token should expire at
     */
    private function setRememberCookie(?string $selector, ?string $token, int $expires)
    {
        $params = \session_get_cookie_params();

        if (isset($selector) && isset($token)) {
            $content = $selector . self::COOKIE_CONTENT_SEPARATOR . $token;
        } else {
            $content = '';
        }

        // save the cookie with the selector and token (requests a cookie to be written on the client)
        $cookie = new Cookie($this->rememberCookieName);
        $cookie->setValue($content);
        $cookie->setExpiryTime($expires);
        $cookie->setPath($params['path']);
        $cookie->setDomain($params['domain']);
        $cookie->setHttpOnly($params['httponly']);
        $cookie->setSecureOnly($params['secure']);
        $result = $cookie->save();

        if ($result === false) {
            throw new HeadersAlreadySentError();
        }

        // if we've been deleting the cookie above
        if (!isset($selector) || !isset($token)) {
            // attempt to delete a potential old cookie from versions v1.x.x to v6.x.x as well (requests a cookie to be written on the client)
            $cookie = new Cookie('auth_remember');
            $cookie->setPath((!empty($params['path'])) ? $params['path'] : '/');
            $cookie->setDomain($params['domain']);
            $cookie->setHttpOnly($params['httponly']);
            $cookie->setSecureOnly($params['secure']);
            $cookie->delete();
        }
    }

    /**
     * @param int $userId
     * @param string $loginName
     * @param int $status
     * @param int $role
     * @param int $forceLogout
     * @param bool $remembered
     */
    protected function onLoginSuccessful(int $userId, string $loginName, int $status, int $role, int $forceLogout, bool $remembered = false)
    {
        // обновить метку времени последнего входа пользователя
        connect('users')->where('userId', $userId)->update(['lastLogin' => gmdate('Y-m-d H:i:s')]);

        parent::onLoginSuccessful($userId, $loginName, $status, $role, $forceLogout, $remembered);
    }

    /**
     * Удаляет session cookie на клиенте.
     *
     * @throws HeadersAlreadySentError
     */
    private function deleteSessionCookie()
    {
        $params = \session_get_cookie_params();

        // ask for the session cookie to be deleted (requests a cookie to be written on the client)
        $cookie = new Cookie(\session_name());
        $cookie->setPath($params['path']);
        $cookie->setDomain($params['domain']);
        $cookie->setHttpOnly($params['httponly']);
        $cookie->setSecureOnly($params['secure']);
        $result = $cookie->delete();

        if ($result === false) {
            throw new HeadersAlreadySentError();
        }
    }


    /**
     * Аутентифиция существующего пользователя
     *
     * @param  string|null  $password  phone пользователя
     * @param  string|null  $phone  (optional) phone пользователя
     * @param  string|null  $loginName (optional) loginName пользователя
     * @param  string       $primaryKey  поле (phone, loginName) пользователя
     * @param  bool   $rememberDuration  (optional) время в секундах, в течение которого пользователь остается в системе («запомнить меня»), например `60 * 60 * 24 * 365,25` на один год
     * @param  callable|null $onBeforeSuccess  (optional) функция, которая получает ID пользователя в качестве единственного параметра и выполняется до успешной аутентификации; должен вернуть true для продолжения или false для отмены
     *
     * @throws EmailNotVerifiedException если email address еще не был подтвержден по email с подтверждением
     * @throws InvalidPasswordException если password был недействителен
     * @throws TooManyRequestsException если количество разрешенных попыток / запросов было превышено
     * @throws UnknownUsernameException Пользователь не найден.
     * @throws UsernameRequiredError если email address недействителен или не может быть найден
     * @throws UsernameBlockedException
     * @throws UnknownIdException
     */
    private function authenticate(?string $password, ?string $phone, string $loginName, string $primaryKey, bool $rememberDuration = false, callable $onBeforeSuccess = null)
    {
        $this->throttle(['enumerateUsers', $this->getIpAddress()], 1, (60 * 60), 75);
        $this->throttle(['attemptToLogin', $this->getIpAddress()], 4, (60 * 60), 5, true);

        // Validate login credentials
        $login = $this->getLogin($phone, $loginName);

        if ($this->blockHacking) {
            // Метод увеличения количества попыток
            $this->incrementLoginAttempts($login);

            // Проверить количество попыток
            $this->checkLoginAttempts();
        }

        // Возвращает данные пользователя для учетной записи с указанным именем входа (если есть)
        $userData = $this->getUserDataByUsername($primaryKey, $login);

        // Валидация пароля
        $this->validateAndVerifyPassword($userData, $password, $primaryKey, $login);

        // Проверка на верификацию email
        $this->checkEmailVerification($userData);

        // Метод, который получает ID пользователя в качестве единственного параметра и выполняется до успешной аутентификации;
        if ($this->handleLoginSuccess($userData, $rememberDuration, $onBeforeSuccess)) {
            return;
        }

        // Обработка отмены аутентификации
        $this->handleFailedLogin($primaryKey, $login);

    }

    /**
     * Проверка учетных данных для входа в систему
     * @throws UsernameRequiredError
     */
    private function getLogin(?string $phone, ?string $loginName): string
    {
        if (!empty($phone)) {
            return \Mnv\Core\Utilities\Mask::init()->phone($phone);
        } elseif (!empty($loginName)) {
            return \trim($loginName);
        }

        throw new UsernameRequiredError();
    }


    /**
     * Валидация пароля
     *
     * @param  string|null  $password
     * @param  string  $primaryKey
     * @param  array  $userData
     *
     * @return void
     *
     * @throws InvalidPasswordException
     * @throws TooManyRequestsException
     * @throws UnknownIdException
     * @throws ValidatePasswordException
     */
    private function validateAndVerifyPassword(array $userData, ?string $password, string $primaryKey, string $login)
    {
        // Валидация пароля
        $password = self::validatePassword($password);
        if (!\password_verify($password, $userData['password'])) {
            $this->handleFailedLogin($primaryKey, $login);
            // мы не можем аутентифицировать пользователя из-за неправильного пароля
            $this->logFailedAttempt(UserInfo::get_ip(), "Login : $login, Invalid password : $password");
            throw new InvalidPasswordException();
        }

        // Обновление пароля, если необходимо
        if (\password_needs_rehash($userData['password'], \PASSWORD_DEFAULT)) {
            $this->updatePasswordInternal($userData['userId'], $password);
        }
    }

    /**
     * Проверка на верификацию email
     *
     * @param  array  $userData
     *
     * @return void
     * @throws EmailNotVerifiedException
     */
    private function checkEmailVerification(array $userData)
    {
        if ($userData['verified'] != 1) {
            throw new EmailNotVerifiedException();
        }
    }

    /**
     * Метод, который получает ID пользователя в качестве единственного параметра и выполняется до успешной аутентификации;
     *
     * @param  array  $userData
     * @param  bool  $rememberDuration
     * @param  callable|null  $onBeforeSuccess
     *
     * @return bool должен вернуть `true` для продолжения или `false` для отмены
     */
    private function handleLoginSuccess(array $userData, bool $rememberDuration, ?callable $onBeforeSuccess): bool
    {
        if (!isset($onBeforeSuccess) || (is_callable($onBeforeSuccess) && $onBeforeSuccess($userData['userId']) === true)) {
            $this->onLoginSuccessful($userData['userId'], $userData['loginName'], $userData['status'], $userData['groupId'], $userData['forceLogout'], false);

            if ($rememberDuration !== false) {
                $this->createRememberDirective($userData['userId'], 60 * 60 * 24 * 365.25);
            }

            $this->resetLoginAttempts();

            return true;
        }

        return false;
    }

    // Метод для обработки неудачной попытки входа

    /**
     * Обработка блокировки пользователя при большом кол-ве запросоа
     * @param  string  $login
     * @param  string  $primaryKey
     *
     * @return void
     * @throws TooManyRequestsException
     */
    private function handleFailedLogin(string $primaryKey, string $login)
    {
        $this->throttle(['attemptToLogin', $this->getIpAddress()], 4, 60 * 60, 5, false);

        $this->throttle(['attemptToLogin', $primaryKey, $login], 500, (60 * 60 * 24), null, false);

    }


    /**
     * Проверка количества попыток
     *
     * @throws UsernameBlockedException
     */
    public function checkLoginAttempts()
    {
        $userAttempts = $this->getUserAttempts();
        $now = time();

        // Проверка на блокировку пользователя
        if ($userAttempts && $userAttempts['blocked_until'] && strtotime($userAttempts['blocked_until']) > $now) {
            $remainingTime = strtotime($userAttempts['blocked_until']) - $now;
            throw new UsernameBlockedException(lang('login:blocked_until') . " " . gmdate("i:s", $remainingTime) . ".");
        }

        // Повторите попытку через несколько секунд.
//        if ($userAttempts && $userAttempts['attempts'] >= $this->warningThreshold && strtotime($userAttempts['last_attempt']) + $this->retryDelay > $now) {
//            throw new UsernameBlockedException(lang('login:retry_delay'));
//        }

        // Блокировка после превышения количества попыток - Вы заблокированы на :time из-за слишком большого количества неудачных попыток входа в систему.
        if ($userAttempts && $userAttempts['attempts'] >= $this->maxAttempts) {
            $this->blockUser();

            $formattedTime = $this->formatBlockTime(($this->blockDuration / 60));
            $block_duration_text = str_replace(':time', $formattedTime, lang('login:block_duration'));

            // 'You are blocked on ' . ($this->blockDuration / 60) . ' minutes due to a lot of failed attempts'
            throw new UsernameBlockedException($block_duration_text);
        }
    }

    private function formatBlockTime($minutes): string
    {
        if ($minutes >= 60) {
            $hours = floor($minutes / 60);
            return $hours . ' ' . pluralLanguage($hours, 'hours');
        } else {
            return $minutes . ' ' . pluralLanguage($minutes, 'minutes');
        }
    }


    /**
     * Возвращает, вошел ли пользователь в систему, читая данные из session.
     *
     * @return boolean whether the user is logged in or not
     */
    public function isLoggedIn(): bool
    {
        return isset($_SESSION[self::SESSION_FIELD_LOGGED_IN]) && $_SESSION[self::SESSION_FIELD_LOGGED_IN] === true;
    }

    /**
     * Сокращение / псевдоним для ´isLoggedIn()´
     *
     * @return boolean
     */
    public function check(): bool
    {
        return $this->isLoggedIn();
    }

    /**
     * Возвращает идентификатор пользователя, вошедшего в систему, путем чтения из сеанса
     *
     * @return int the user ID
     */
    public function getUserId(): ?int
    {
        return $_SESSION[self::SESSION_FIELD_USER_ID] ?? null;
    }

    /**
     * Сокращение / псевдоним для {@see getUserId}
     *
     * @return int
     */
    public function id(): ?int
    {
        return $this->getUserId();
    }


    /**
     * Возвращает статус текущего вошедшего в систему пользователя из session
     *
     * @return int the status as one of the constants from the {@see Status} class
     */
    public function getStatus(): ?int
    {
        return $_SESSION[self::SESSION_FIELD_STATUS] ?? null;
    }

    /**
     * Возвращает, находится ли текущий авторизованный пользователь в "нормальном" состоянии.
     * Returns whether the currently signed-in user is in "normal" state
     *
     * @return bool
     *
     * @see Status
     * @see Administration::getStatus
     */
    public function isNormal(): bool
    {
        return $this->getStatus() === Status::NORMAL;
    }

    /**
     * Возвращает, находится ли текущий авторизованный пользователь в "заблокированном" состоянии.
     * Returns whether the currently signed-in user is in "locked" state
     *
     * @return bool
     *
     * @see Status
     * @see Administration::getStatus
     */
    public function isLocked(): bool
    {
        return $this->getStatus() === Status::PENDING_REVIEW;
    }


    /**
     * Returns the user's current IP address
     *
     * @return string the IP address (IPv4 or IPv6)
     */
    public function getIpAddress(): ?string
    {
        return $this->ipAddress;
    }

    /**
     * Возвращает, запомнился ли текущий авторизованный пользователь долгоживущим файлом cookie.
     * Returns whether the currently signed-in user has been remembered by a long-lived cookie
     *
     * @return bool whether they have been remembered
     */
    public function isRemembered(): ?bool
    {
        return $_SESSION[self::SESSION_FIELD_REMEMBERED] ?? null;
    }


    public function deviceId()
    {
        $this->deviceId = $_SESSION[self::SESSION_FIELD_DEVICE] ?? null;
    }

    public function admin(): ?array
    {
        return $_SESSION ?? [];
    }

    public function limit(): int
    {
        if (isset($_SESSION[self::SESSION_FIELD_DEVICE])) {
            $deviceId = $_SESSION[self::SESSION_FIELD_DEVICE];

            $userAttempts = connect('user_login_attempts')->select('attempts')->where('deviceId', $deviceId)->get('array');
            if (!empty($userAttempts)) {
                return $userAttempts['attempts'];
            }
        }

        return $this->maxAttempts;
    }



    /**
     * Возвращает, имеет ли текущий вошедший в систему пользователь указанную роль.
     *
     * @param string $role
     * @return bool
     */
    public function hasRole(string $role): bool
    {
        if (empty($role)) {
            return false;
        }

        if (isset($_SESSION[self::SESSION_FIELD_ROLES])) {

            $role = (int) $role;

            return (((int) $_SESSION[self::SESSION_FIELD_ROLES]) & $role) === $role;
        } else {
            return false;
        }
    }

    /**
     * Возвращает true, имеет ли текущий вошедший в систему пользователь * любую * из указанных ролей.
     *
     * @param array $roles
     * @return bool
     */
    public function hasAnyRole(array $roles): bool
    {
        foreach ($roles as $role) {
            if ($this->hasRole($role)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Возвращает true, есть ли у текущего вошедшего в систему пользователя * все * указанные роли.
     * Returns whether the currently signed-in user has *all* of the specified roles
     *
     * @param string[] ...$roles
     * @return bool
     */
    public function hasAllRoles(...$roles): bool
    {
        foreach ($roles as $role) {
            if (!$this->hasRole($role)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Возвращает, имеет ли пользователь с данным идентификатором указанную роль
     *
     * @param int $userId the ID of the user to check the roles for
     * @param array $roles
     * @return bool
     * @throws UnknownIdException if no user with the specified ID has been found
     */
    public function doesUserHaveRole(int $userId, array $roles): bool
    {
        if (empty($roles)) {
            return false;
        }
        $groupId = connect('users AS u')
            ->select('u.groupId')
            ->leftJoin('user_group AS g', 'g.groupId', '=', 'u.groupId')
            ->where('g.hasAdminAccess', 1) // todo: можно переделать к каждому пользователю
            ->where('u.userId', $userId)
            ->in('u.groupId', $roles)
            ->getValue();

        if ($groupId === null) {
            throw new UnknownIdException();
        }

        return true;
    }


    /**
     *  Метод `throttle` реализует алгоритм регулирования или ограничения скорости запросов с использованием алгоритма "маркерная корзина" (Token Bucket).
     *
     * @param array $criteria отдельные критерии, которые вместе описывают регулируемый ресурс
     * @param int $supply количество единиц, предоставляемых за интервал (>= 1)
     * @param int $interval интервал (в секундах), на который предоставляется подача (>= 5)
     * @param int|null $impetuosity (optional) допустимая степень вариации или неравномерности во время пиков (>= 1)
     * @param bool|null $simulated (optional) следует ли имитировать пробный запуск вместо фактического потребления запрошенных единиц
     * @param int|null $cost (optional) количество единиц для запроса (>= 1)
     * @param bool|null $force (optional) применять ли регулирование локально (с помощью этого вызова), даже если регулирование было отключено глобально (на экземпляре с помощью параметра конструктора)
     * @throws TooManyRequestsException если фактический спрос превысил обозначенное предложение
     */
    public function throttle(array $criteria, int $supply, int $interval, int $impetuosity = null, bool $simulated = null, int $cost = null, bool $force = null)
    {
        // проверьте предоставленные параметры и при необходимости установите соответствующие значения по умолчанию
        $force = $force ?? false;
        $impetuosity = $impetuosity ?? 1000;
        $simulated = $simulated ?? false;
        $cost = $cost ?? 1;

        if (!$this->throttling && !$force) {
            return $supply;
        }

        // сгенерировать уникальный ключ для корзины (состоящий из 44 или менее символов ASCII)
        $key = Base64::encodeUrlSafeWithoutPadding(\hash('sha256', \implode("\n", $criteria), true));
        $now = \time();

        // Рассчет пропускной способности и емкости корзины
        $capacity = $impetuosity * $supply;
        $bandwidthPerSecond = $supply / $interval;

        $bucket = connect('users_throttling')->select('tokens, replenished_at')->where('bucket', $key)->get('array') ?? [];

        // Инициализация корзины
        $bucket['tokens'] = $bucket['tokens'] ?? $capacity;
        // инициализировать время последнего пополнения корзины (как временная метка Unix в секундах)
        $bucket['replenished_at'] = $bucket['replenished_at'] ?? $now;

        // Пополнение корзины
        $bucket = $this->replenishBucket($bucket, $capacity, $bandwidthPerSecond, $now);

        // Проверка возможности выполнения запроса
        if ($bucket['tokens'] >= $cost) {
            if (!$simulated) {
                // Сохранение обновленных данных корзины
                $bucket['tokens'] = max(0, $bucket['tokens'] - $cost);
                $this->updateBucket($bucket, $key, $capacity, $bandwidthPerSecond, $now);
            }
            return $bucket['tokens'];
        } else {
            // Блокировка пользователя и выброс исключения
            $this->tooManyBlockUser();
            $tokensMissing = $cost - $bucket['tokens'];
            $estimatedWaitingTimeSeconds = ceil($tokensMissing / $bandwidthPerSecond);
            throw new TooManyRequestsException('', $estimatedWaitingTimeSeconds);
        }
    }

    /**
     * Пополняет корзину токенами.
     *
     * @param array $bucket данные корзины
     * @param float $capacity емкость корзины
     * @param float $bandwidthPerSecond скорость пополнения корзины в секунду
     * @param int $now текущее время
     * @return array обновленные данные корзины
     */
    private function replenishBucket(array $bucket, float $capacity, float $bandwidthPerSecond, int $now): array
    {
        $secondsSinceLastReplenishment = max(0, $now - $bucket['replenished_at']);
        $tokensToAdd = $secondsSinceLastReplenishment * $bandwidthPerSecond;
        $bucket['tokens'] = min($capacity, $bucket['tokens'] + $tokensToAdd);
        $bucket['tokens'] = str_replace(',', '.', (string)$bucket['tokens']);
        $bucket['replenished_at'] = $now;

        return $bucket;
    }

    /**
     * Обновляет данные корзины в базе данных.
     *
     * @param array $bucket данные корзины
     * @param string $key ключ корзины
     * @param float $capacity емкость корзины
     * @param float $bandwidthPerSecond скорость пополнения корзины
     * @param int $now текущее время
     */
    private function updateBucket(array $bucket, string $key, float $capacity, float $bandwidthPerSecond, int $now): void
    {
        $bucket['bucket'] = $key;
        $bucket['tokens'] = str_replace(',', '.', $bucket['tokens']);
        $bucket['expires_at'] = $now + floor($capacity / $bandwidthPerSecond * 2);

        // Начать транзакцию для безопасности
//        connect()->transaction();
//        try {
            // Вставка или обновление данных с помощью UPSERT
            connect('users_throttling')->upsert($bucket);

//            connect()->commit();
//        } catch (\Exception $e) {
//            connect()->rollBack();
//            // Логирование ошибок
//            error_log('Error updating token bucket: ' . $e->getMessage());
//        }
    }


    /**
     * @return bool
     */
    public function isRememberCookieName(): bool
    {
        return isset($_COOKIE[$this->rememberCookieName]);
    }

    /**
     * Generates a unique cookie name for the given descriptor based on the supplied seed
     *
     * @param string $descriptor a short label describing the purpose of the cookie, e.g. 'session'
     * @param string|null $seed (optional) the data to deterministically generate the name from
     * @return string
     */
    public static function createCookieName(string $descriptor, ?string $seed = null): string
    {
        // use the supplied seed or the current UNIX time in seconds
        $seed = ($seed !== null) ? $seed : \time();

        foreach (self::COOKIE_PREFIXES as $cookiePrefix) {
            // if the seed contains a certain cookie prefix
            if (\strpos($seed, $cookiePrefix) === 0) {
                // prepend the same prefix to the descriptor
                $descriptor = $cookiePrefix . $descriptor;
            }
        }

        // generate a unique token based on the name(space) of this library and on the seed
        $token = Base64::encodeUrlSafeWithoutPadding(\md5(__NAMESPACE__ . "\n" . $seed, true));

        return $descriptor . '_' . $token;
    }

    /**
     * Generates a unique cookie name for the 'remember me' feature
     * Создает уникальное имя файла cookie для функции «запомнить меня»
     *
     * @param string|null $sessionName (optional) session name на котором должен быть основан вывод
     * @return string
     */
    public static function createRememberCookieName(?string $sessionName = null): string
    {
        return self::createCookieName('remember_mnv', ($sessionName !== null) ? $sessionName : \session_name());
    }

    /**
     * Returns the selector of a potential locally existing remember directive
     * Возвращает селектор потенциально существующей локально директивы запоминания.
     *
     * @return string|null
     */
    private function getRememberDirectiveSelector(): ?string
    {
        if (isset($_COOKIE[$this->rememberCookieName])) {
            $selectorAndToken = \explode(self::COOKIE_CONTENT_SEPARATOR, $_COOKIE[$this->rememberCookieName], 2);

            return $selectorAndToken[0];
        } else {
            return null;
        }
    }

    /**
     * Returns the expiry date of a potential locally existing remember directive
     *
     * @return int|null
     */
    private function getRememberDirectiveExpiry(): ?int
    {
        // если пользователь в настоящее время вошел в систему
        if ($this->isLoggedIn()) {
            // определить селектор любой существующей в настоящее время директивы запоминания
            $existingSelector = $this->getRememberDirectiveSelector();

            // если в настоящее время существует директива запоминания, селектор которой мы только что получили
            if (isset($existingSelector)) {
                // получить дату истечения срока действия для данного селектора
                $existingExpiry = connect('users_remembered')->select('expires')->where('selector',  $existingSelector)->where('user',  $this->getUserId())->get('array');
                // если установлен срок годности
                if (!empty($existingExpiry)) {
                    return (int) $existingExpiry['expires'];
                }
            }
        }

        return null;
    }

}