<?php

namespace Mnv\Core;

/**
 * BotBlockIp
 *
 * @package Mnv\Core
 */
class BotBlockIp
{

    /** Время блокировки в секундах. */
    const BLOCK_SECONDS = 60;

    /** Интервал времени запросов страниц в секундах. */
    const INTERVAL_SECONDS = 0.5;

    /** Максимальное количество запросов страницы в интервал времени. */
    const MAX_REQUESTS = 4;

    /** Путь к директории кэширования активных пользователей. */
    const PATH_ACTIVE = 'temp/active';

    /** Путь к директории кэширования заблокированных пользователей. */
    const PATH_BLOCKED = 'temp/block';

    /** Флаг абсолютных путей к директориям. */
    const PATH_IS_ABSOLUTE = false;

    /** Флаг подключения всегда активных пользователей. */
    const isAlwaysActive = true;

    /** Флаг подключения всегда заблокированных пользователей. */
    const isAlwaysBlock = true;

    /** @var int|mixed|string */
    private static $timeBlocked = '';


    /** Список всегда активных пользователей. */
    public static array $alwaysActive = array(
//        '127.0.0.1'
    );

    /** Список всегда заблокированных пользователей. */
    public static array $alwaysBlock = array(
//        '127.0.0.1'
    );


    /**
     * Метод проверки ip-адреса на активность и блокировку.
     */
    public static function checkIp()
    {
        // Если это поисковый бот, то выходим ничего не делая
        if (self::isBot()) {
            return;
        }

        // Получение ip-адреса
        $ipAddress = self::_getIp();

        // Пропускаем всегда активных пользователей
        if (in_array($ipAddress, self::$alwaysActive) && self::isAlwaysActive) {
            return;
        }

        // Блокируем всегда заблокированных пользователей
        if (in_array($ipAddress, self::$alwaysBlock) && self::isAlwaysBlock) {
            self::sendBlockedResponse403();

        }

        // Установка путей к директориям
        $pathActive = self::getAbsolutePath(self::PATH_ACTIVE);
        $pathBlocked = self::getAbsolutePath(self::PATH_BLOCKED);

        // Проверка возможности записи в директории
        self::checkDirectoryWritable($pathActive, 'Директория кэширования активных пользователей недоступна для записи.');
        self::checkDirectoryWritable($pathBlocked, 'Директория кэширования заблокированных пользователей недоступна для записи.');

        // Проверка активных ip-адресов
        $isActive = self::checkActiveIps($pathActive, $ipAddress);
        // Проверка заблокированных ip-адресов
        $isBlocked = self::checkBlockedIps($pathBlocked, $ipAddress);

        // ip-адрес заблокирован
        if ($isBlocked) {
            self::sendBlockedResponse502();
        }

        // Создание идентификатора активного ip-адреса
        if (!$isActive) {
            touch($pathActive . $ipAddress . '_' . time());
        }
    }

    /** Метод получения текущего ip-адреса из переменных сервера. */
    private static function _getIp(): string
    {
        // ip-адрес по умолчанию
        $ipAddress = '127.0.0.1';

        // Массив возможных ip-адресов
        $address = [];

        // Сбор данных возможных ip-адресов
        if (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            // Проверяется массив ip-клиента установленных прозрачными прокси-серверами
            foreach (array_reverse(explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])) as $value) {
                $value = trim($value);
                // Собирается ip-клиента
                if (preg_match('#^\d{1,3}.\d{1,3}.\d{1,3}.\d{1,3}$#', $value)) {
                    $address[] = $value;
                }
            }
        }

        // Сбор возможных IP-адресов
        foreach (['HTTP_X_FORWARDED_FOR', 'HTTP_CLIENT_IP', 'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_PROXY_USER', 'REMOTE_ADDR'] as $header) {
            if (!empty($_SERVER[$header])) {
                $address = array_merge($address, explode(',', $_SERVER[$header]));
            }
        }

        // Фильтрация возможных ip-адресов, для выявления нужного
        foreach ($address as $value) {
            // Выбирается ip-клиента
            if (preg_match('#^(\d{1,3}).(\d{1,3}).(\d{1,3}).(\d{1,3})$#', $value, $matches)) {
                $value = $matches[1] . '.' . $matches[2] . '.' . $matches[3] . '.' . $matches[4];
                if ('...' != $value) {
                    $ipAddress = $value;
                    break;
                }
            }
        }

        // Возврат полученного ip-адреса
        return $ipAddress;
    }

    /** Метод проверки на поискового бота. */
    private static function isBot(): bool
    {
        if (!empty($_SERVER['HTTP_USER_AGENT'])) {
            $options = array(
                'YandexBot', 'YandexAccessibilityBot', 'YandexMobileBot','YandexDirectDyn',
                'YandexScreenshotBot', 'YandexImages', 'YandexVideo', 'YandexVideoParser',
                'YandexMedia', 'YandexBlogs', 'YandexFavicons', 'YandexWebmaster',
                'YandexPagechecker', 'YandexImageResizer','YandexAdNet', 'YandexDirect',
                'YaDirectFetcher', 'YandexCalendar', 'YandexSitelinks', 'YandexMetrika',
                'YandexNews', 'YandexNewslinks', 'YandexCatalog', 'YandexAntivirus',
                'YandexMarket', 'YandexVertis', 'YandexForDomain', 'YandexSpravBot',
                'YandexSearchShop', 'YandexMedianaBot', 'YandexOntoDB', 'YandexOntoDBAPI',
                'Googlebot', 'Googlebot-Image', 'Mediapartners-Google', 'AdsBot-Google',
                'Mail.RU_Bot', 'bingbot', 'Accoona', 'ia_archiver', 'Ask Jeeves',
                'OmniExplorer_Bot', 'W3C_Validator', 'WebAlta', 'YahooFeedSeeker', 'Yahoo!',
                'Ezooms', '', 'Tourlentabot', 'MJ12bot', 'AhrefsBot', 'SearchBot', 'SiteStatus',
                'Nigma.ru', 'Baiduspider', 'Statsbot', 'SISTRIX', 'AcoonBot', 'findlinks',
                'proximic', 'OpenindexSpider','statdom.ru', 'Exabot', 'Spider', 'SeznamBot',
                'oBot', 'C-T bot', 'Updownerbot', 'Snoopy', 'heritrix', 'Yeti',
                'DomainVader', 'DCPbot', 'PaperLiBot'
            );

            foreach($options as $row) {
                if (stripos($_SERVER['HTTP_USER_AGENT'], $row) !== false) {
                    return true;
                }
            }
        }

        return false;
    }

    /** Проверка директории на доступность для записи. */
    private static function checkDirectoryWritable(string $path, string $errorMessage)
    {
        if (!is_writable($path)) {
            die($errorMessage);
        }
    }

    /** Проверка активных IP-адресов. */
    private static function checkActiveIps(string $path, string $ipAddress): bool
    {
        if ($dir = opendir($path)) {
            while (false !== ($filename = readdir($dir))) {
                if (preg_match('#^(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})_(\d+)$#', $filename, $matches)) {
                    if ($matches[1] === $ipAddress && $matches[2] >= time() - self::INTERVAL_SECONDS) {
                        $times = intval(file_get_contents($path . '/' . $filename));
                        if ($times >= self::MAX_REQUESTS - 1) {
                            // Блокируем IP
                            touch(self::getAbsolutePath(self::PATH_BLOCKED) . '/' . $filename);
                            unlink($path . '/' . $filename);
                        } else {
                            file_put_contents($path . '/' . $filename, $times + 1);
                        }
                        closedir($dir);
                        return true; // IP найден как активный
                    } elseif ($matches[2] < time() - self::INTERVAL_SECONDS) {
                        unlink($path . '/' . $filename); // Удаляем старые записи
                    }
                }
            }
            closedir($dir);
        }
        return false; // IP не найден
    }

    /** Проверка заблокированных IP-адресов. */
    private static function checkBlockedIps(string $path, string $ipAddress): bool
    {
        if ($dir = opendir($path)) {
            while (false !== ($filename = readdir($dir))) {
                if (preg_match('#^(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})_(\d+)$#', $filename, $matches)) {
                    if ($matches[1] === $ipAddress && $matches[2] >= time() - self::BLOCK_SECONDS) {
                        closedir($dir);
                        self::$timeBlocked = $matches[2] - (time() - self::BLOCK_SECONDS) + 1;
                        return true; // IP найден как заблокированный
                    } elseif ($matches[2] < time() - self::BLOCK_SECONDS) {
                        unlink($path . '/' . $filename); // Удаляем старые записи
                    }
                }
            }
            closedir($dir);
        }
        return false; // IP не найден
    }


    /** 403 Метод отправки ответа о блокировке. */
    private static function sendBlockedResponse403()
    {
        header('HTTP/1.0 403 Forbidden');
        echo '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">';
        echo '<html xmlns="http://www.w3.org/1999/xhtml">';
        echo '<head>';
        echo '<title>Вы заблокированы</title>';
        echo '<meta http-equiv="content-type" content="text/html; charset=utf-8" />';
        echo '</head>';
        echo '<body>';
        echo '<p style="background:#ccc;border:solid 1px #aaa;margin:30px auto;padding:20px;text-align:center;">';
        echo 'Вы заблокированы администрацией ресурса.<br />';
        echo '</p>';
        echo '</body>';
        echo '</html>';
        exit;
    }

    /** 502 Метод отправки ответа о блокировке. */
    private static function sendBlockedResponse502()
    {
        header('HTTP/1.0 502 Bad Gateway');
        echo '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">';
        echo '<html xmlns="http://www.w3.org/1999/xhtml">';
        echo '<head>';
        echo '<title>502 Bad Gateway</title>';
        echo '<meta http-equiv="content-type" content="text/html; charset=utf-8" />';
        echo '</head>';
        echo '<body>';
        echo '<h1 style="text-align:center">502 Bad Gateway</h1>';
        echo '<p style="background:#ccc;border:solid 1px #aaa;margin:30px auto;padding:20px;text-align:center;">';
        echo 'К сожалению, Вы временно заблокированы, из-за частого запроса страниц сайта. <br />Вам придется подождать. Через ' . self::$timeBlocked . ' секунд(ы) Вы будете автоматически разблокированы.';
        echo '</p>';
        echo '</body>';
        echo '</html>';
        exit;
    }


    /**
     * Получение абсолютного пути к директории.
     */
    private static function getAbsolutePath(string $relativePath): string
    {
        return self::PATH_IS_ABSOLUTE
            ? $relativePath
            : str_replace('\\', '/', GLOBAL_ROOT . '/' . rtrim($relativePath, '/') . '/');
    }


}