<?php

namespace Mnv\Core;

use Mnv\Core\Singleton\SingletonTrait;

/**
 * Class Config
 * @package Mnv
 */
class ConfigManager
{
    use SingletonTrait;

    /** @var array|null $_config Сохраняет настройки, загруженные из базы данных */
    private ?array $_config = null;

    /** @var array $_buffer Сохраняет кэшированные настройки  */
    private array $_buffer = [];

    /** Config constructor. */
    public function __construct()
    {
        // Инициализация может быть отложена до тех пор, пока не будет запрошена настройка
    }

    /**
     * Получение всех настроек
     * @return array
     */
    public function config(): array
    {
        // Загружайте настройки только в том случае, если они еще не были загружены
        if ($this->_config === null) {
            $this->_config = connect('settings')->select('codename, value')->pluck('value', 'codename');
        }

        return $this->_config;
    }

    /**
     * Установка значения в конфигурации и буфере
     *
     * @param string $codename
     * @param string $value
     */
    public function set(string $codename, string $value): void
    {
        // Загружаем, конфигурацию
        $this->config();
        $this->_config[$codename] = $value;
        // Кэшируем обновленное значение
        $this->_buffer[$codename] = $value;
    }

    /**
     * Получение значения настройки по ключу
     * @param string $codename
     * @return mixed|null
     */
    public function get(string $codename)
    {
        // Проверьте, существует ли это значение в буфере
        if (isset($this->_buffer[$codename])) {
            return $this->_buffer[$codename];
        }

        // Загружаем настройки, если они еще не загружены
        $this->config();

        // Если его нет в буфере, извлекаем значение из конфигурации и буферизуем его
        if (isset($this->_config[$codename])) {
            $this->_buffer[$codename] = $this->_config[$codename];
            return $this->_config[$codename];
        }

        // Вызывает ошибку, если ключа не существует
        $trace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 1);
        trigger_error(
            'Неопределённое свойство: ' . $codename . ' в ' . $trace[0]['file'] . ' на строке ' . $trace[0]['line'],
            E_USER_NOTICE
        );
        return null;
    }

    /**
     * Получение значения настройки (статический вызов)
     * @param string $codename
     * @return mixed|null
     */
    public static function getValue(string $codename)
    {
        return self::init()->get($codename);
    }
}


