<?php

namespace Mnv\Core;

use Mnv\Core\Bot\Telegram;
use Mnv\Core\Bot\TelegramNotifications;
use Mnv\Core\Filesystem\FilesystemManager;
use Mnv\Core\Locale\I18N;
use Mnv\Core\Test\Log;
use Mnv\Core\Uploads\Uploader;
use Mnv\Core\Utilities\GUID;
use Mnv\Models\UserGroups;
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;
use Symfony\Component\HttpFoundation\File\UploadedFile as SymfonyUploadedFile;

/**
 * Class Feedback
 * @package Mnv\Core
 */
class Feedback
{
    private $charset;
    private $smtp_server;
    private $smtp_mail;
    private $smtp_user;
    private $smtp_pass;
    private $smtp_secure;
    private $smtp_port;
    private $smtp_debug;

    private $remoteIp;
    private $private_key;

    public $errors;
    public $message;
    public $success = [];

//    private $type;
    private $captchaEnabled;
    private $allowCaptcha;

    protected $companyName;
    protected $recipient;

    private $telegramConfig;

    private $messenger;

    protected $feedbackId;

    /** @var array  */
    private array $uploaded_file = [];

    /**
     * Feedback constructor.
     */
    public function __construct()
    {

        $this->telegramConfig = TelegramNotifications::init()->config();

        $this->charset     = ConfigManager::getValue('charset');
        $this->smtp_debug  = ConfigManager::getValue('smtp_debug');
        $this->smtp_server = ConfigManager::getValue('smtp_server');
        $this->smtp_mail   = ConfigManager::getValue('smtp_mail');
        $this->smtp_user   = ConfigManager::getValue('smtp_user');
        $this->smtp_pass   = ConfigManager::getValue('smtp_pass');
        $this->smtp_secure = ConfigManager::getValue('smtp_secure');
        $this->smtp_port   = ConfigManager::getValue('smtp_port');

        $this->companyName =  ConfigManager::getValue('company_name');
        $this->recipient = ConfigManager::getValue('email');
        $this->remoteIp    = $_SERVER['REMOTE_ADDR'];

        $this->captchaEnabled = ConfigManager::getValue('feedback_captcha_enabled');
        $this->allowCaptcha = ConfigManager::getValue('allow_recaptcha');
        $this->private_key = ConfigManager::getValue('recaptcha_private_key');

//        print_r($this);
    }


    public function getCaptchaEnabled(): int
    {
        return $this->captchaEnabled;
    }

    public function getAllowCaptcha(): int
    {
        return $this->allowCaptcha;
    }

    /**
     * Метод для проверки длины строки
     * @param $string
     * @param $minLength
     * @param $maxLength
     * @return bool
     */
    public function checkStringLength($string, $minLength, $maxLength): bool
    {
        $length = mb_strlen($string, ConfigManager::getValue('charset'));
        return !(($length < $minLength) || ($length > $maxLength));
    }

    /**
     * Метод проверки имени (name)
     *
     * @param $name
     * @return bool
     */
    public function checkName($name): bool
    {
        if (!empty($name)) {
            if (!$this->checkStringLength($name, 2, 30)) {
                $this->errors = I18N::locale(
                    "* Поля имя содержит недопустимое количество символов!",
                    "* Maydon nomi belgilar nomaqbul sonini o'z ichiga oladi!",
                    "* The name field contains an invalid number of characters!");;
                return false;
            }
        } else {
            $this->errors =  I18N::locale(
                "* Не допустимо оставлять поле имя пустым!",
                "* Maydon nomini bo'sh qoldirish joiz emas!",
                "* It is unacceptable to leave the name field empty!"
            );
            return false;
        }
        return true;
    }



    /**
     * проверка капча (reCaptcha:v2)
     *
     * @param $recaptcha_response
     * @return bool
     */
    public function validateReCaptchaV2($recaptcha_response): bool
    {
        $recaptcha = new \ReCaptcha\ReCaptcha($this->private_key);
        $resp = $recaptcha->setExpectedHostname($_SERVER['SERVER_NAME'])->verify($recaptcha_response, $this->remoteIp);
//        $response = $resp->toArray();

        if ($resp->isSuccess()) {
            return true;
        } else {
            $this->errors = $resp->getErrorCodes();
            $this->message = I18N::locale(
                "Возникла проблема с вашей отправкой, пожалуйста, перезагрузите страницу и попробуйте еще раз.",
                "Yuborishda muammo bor, iltimos, sahifani qayta yuklang va qaytadan urinib ko'ring.",
                "There is a problem with your submission, please reload the page and try again."
            );

            return false;
        }
    }

    /**
     * проверка капча (reCaptcha:v3)
     *
     * @param $recaptcha_token
     * @return bool
     */
    public function validateReCaptchaV3($recaptcha_token): bool
    {
        $recaptcha = new \ReCaptcha\ReCaptcha($this->private_key);
        $resp = $recaptcha->setExpectedHostname($_SERVER['SERVER_NAME'])->setScoreThreshold(0.5)->verify($recaptcha_token, $this->remoteIp);
//          $response = $resp->toArray();

        if ($resp->isSuccess()) {
            return true;
        } else {
            $this->errors = $resp->getErrorCodes();
            $this->message = I18N::locale(
                "Возникла проблема с вашей отправкой, пожалуйста, перезагрузите страницу и попробуйте еще раз.",
                "Yuborishda muammo bor, iltimos, sahifani qayta yuklang va qaytadan urinib ko'ring.",
                "There is a problem with your submission, please reload the page and try again."
            );

            return false;
        }
    }

    /**
     * Анти бот
     * @param array $bot
     * @return bool
     */
    public function antiBot(array $bot): bool
    {
        if (trim($bot['anti-bot-a']) !== date('Y')) {
            if (trim($bot['anti-bot-a']) !== trim($bot['anti-bot-q'])) {
                $this->errors = "I'm a robot! Picked up a captcha";
                $this->message =  I18N::locale("Я робот! Подобрал капчу", "Men robotman! Oldi captcha", 'I am a robot! Picked up captcha');
            } elseif (empty($bot['anti-bot-q'])) {
                $this->errors = "An empty answer";
                $this->message = I18N::locale("Ошибка: пустой ответ.", "Xato: bo'sh javob.", "Error: empty response.");
            } else {
                $this->errors = "Wrong answer";
                $this->message = I18N::locale("Ошибка: неправильный ответ.", "Xato: noto'g'ri javob.", "Error: wrong answer.");
            }
        } elseif (!empty($bot['anti-email'])) {
            $this->errors = 'Bots go no!';
        } else {

            return true;
        }

        return false;
    }
    /**
     * Отправка письма через PHPMailer (SMTP)
     *
     * @param string|null $user_email
     * @param string|null $user_name
     * @param string $subject
     * @param string $htmlBody
     * @param SymfonyUploadedFile|null $attachment
     * @return bool
     */
    public function send(?string $user_email, ?string $user_name, string $subject, string $htmlBody, SymfonyUploadedFile $attachment = null): bool
    {
        $mail = new PHPMailer(true);

        try {
            // Настройка SMTP
            $this->configureSMTP($mail);

            // Настройка отправителя
            $this->configureSender($mail);

            // Добавление получателей
            $this->addRecipients($mail, $user_email, $user_name);

            // Добавление вложений
            if ($attachment !== null && $attachment->isValid()) {
                $mail->addAttachment($attachment->getPathname(), $attachment->getClientOriginalName());
            }

            // Установка заголовков и тела сообщения
            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body = $htmlBody;

            // Отправка письма
            $mail->send();
            return true;

        } catch (Exception $e) {
            // Разбор ошибок
            $this->message = "Ошибка отправки: " . $e->getMessage();
            $this->errors = $e->getTraceAsString();
            return false;
        }
    }

    /**
     * Настройка SMTP для PHPMailer.
     *
     * @param PHPMailer $mail
     */
    private function configureSMTP(PHPMailer $mail): void
    {
        $mail->CharSet  = $this->charset ?? 'UTF-8';
        $mail->isSMTP();
        $mail->Timeout = 10;
        $mail->SMTPDebug = $this->smtp_debug;
        $mail->Host = $this->smtp_server;
        $mail->Port = $this->smtp_port;
        $mail->SMTPSecure = $this->smtp_secure;
        $mail->SMTPOptions = $this->getSMTPOptions($this->smtp_secure);

        if (!empty($this->smtp_user)) {
            $mail->SMTPAuth = true;
            $mail->Username = $this->smtp_user;
            $mail->Password = $this->smtp_pass;
        }
    }

    /**
     * Настройка отправителя.
     *
     * @param PHPMailer $mail
     * @throws Exception
     */
    private function configureSender(PHPMailer $mail): void
    {
        $fromEmail = $this->smtp_mail ?: $this->smtp_user;
        $mail->setFrom($fromEmail, $this->companyName);
    }

    /**
     * Получение SMTPOptions в зависимости от протокола
     *
     * @param string|null $secureType
     * @return array
     */
    private function getSMTPOptions(?string $secureType): array
    {
        if (in_array($secureType, ['ssl', 'tls'], true)) {
            return [
                $secureType => [
                    "verify_peer" => false,
                    "verify_peer_name" => false,
                    "allow_self_signed" => true
                ]
            ];
        }
        return [];
    }

    /**
     * Добавление получателей в PHPMailer
     *
     * @param PHPMailer $mail
     * @param string|null $user_email
     * @param string|null $user_name
     * @return void
     * @throws Exception
     */
    private function addRecipients(PHPMailer $mail, ?string $user_email, ?string $user_name): void
    {
        // Основной получатель
        if (!empty($this->recipient)) {
            $mail->addAddress($this->recipient, $this->companyName);
        }

        // Пользователь
        if (!empty($user_email)) {
            $mail->addReplyTo($user_email, $user_name ?: '');
            $mail->addAddress($user_email);
        }

        // Дополнительные получатели (BCC)
        $managers = $this->managersReceiveEmails();
        foreach ($managers as $manager) {
            if (!empty($manager->email)) {
                $mail->addBCC($manager->email, $this->companyName);
            }
        }
    }


    /**
     * Получение менеджеров.
     *
     * @return array
     */
    private function managersReceiveEmails(): array
    {
        return connect('users')->select('email')
            ->in('userType', [UserGroups::DEVELOPER, UserGroups::ADMIN, UserGroups::MANAGER])
            ->where('receiveEmails', 1)
            ->whereNull('email', true)
            ->orderBy('userId', 'ASC')
            ->getAll() ?? [];
    }


    /** *********************** MASK ********************** */


    /**
     * Скрыть несколько симоволов в email
     * @param $email
     * @param int $minLength
     * @param int $maxLength
     * @param string $mask
     * @return string
     */
    public function maskEmail($email, $minLength = 3, $maxLength = 10, $mask = "***"): string
    {
        $atPos = strrpos($email, "@");
        $name = substr($email, 0, $atPos);
        $len = strlen($name);
        $domain = substr($email, $atPos);

        if (($len / 2) < $maxLength) $maxLength = ($len / 2);

        $shortenedEmail = (($len > $minLength) ? substr($name, 0, $maxLength) : "");
        return  "{$shortenedEmail}{$mask}{$domain}";
    }

    public function maskPhone($phone)
    {
        return substr_replace($phone,'****',4,13);
    }


    /** *********************** SAVE ********************** */

    /**
     * @param $feedback
     * @param $htmlBody
     * @param array $data
     * @return void
     */
    public function insertMail($feedback, $htmlBody, array $data): void
    {
        $mails = array_filter([
            'fullName'  => $feedback['fullName'],
            'email'     => $feedback['email'] ?? null,
            'phone'     => $feedback['phone'] ?? null,
            'subject'   => $feedback['subject'],
            'message'   => $htmlBody,
            'data'      => json_encode($data, JSON_UNESCAPED_UNICODE),
            'created'   => gmdate('Y-m-d H:i:s'),

        ]);

        $this->feedbackId = connect('feedback')->insert($mails);

        // сохраняем файл и записываем в базу
        $this->saveFile();
    }

    /**
     * @return void
     */
    public function saveFile()
    {
        if (empty($this->feedbackId)) {
            return null;
        }

        $filesystem = new FilesystemManager('/uploads', 'feedback');

        $uuid = GUID::Format(GUID::Create(), false, '-');
        $upload = new Uploader([], $filesystem->realPath, $filesystem->path, $uuid, 0);
        if ($upload->validate()) {
            if ($upload->uploadFeedback()) {
                if (!empty($upload->response['url'])) {
                    connect('feedback')
                        ->where('id', $this->feedbackId)
                        ->update([
                            'file' => $upload->response['url']
                        ]);

                    return $upload->response['url'];
                }
            }
        }

        return null;
    }

    /** *********************** MESSENGER TELEGRAM ********************** */

    /**
     * @param string $template
     * @param array $data
     *
     * @return void
     */
    public function send_telegram(string $template, array $data = [], $file = []): void
    {
        // Check required conditions before proceeding.
        if (empty($data[0]) || empty($data[1]) || $this->telegramConfig['status'] !== 'V') {
            return;
        }

        if (!empty($file)) {
            $this->populateUploadedFileData($file);
        }

        $this->options($template, $data);
    }
    private function populateUploadedFileData($file): void
    {
        $this->uploaded_file = [
            'name'      => $file->getClientOriginalName(),
            'tmp_name'  => $file->getPathname(),
            'mimeType'  => $file->getClientMimeType(),
            'size'      => $file->getSize(),
            'extension' => $file->getClientOriginalExtension(),
            'error'     => $file->getError(),
        ];
    }

    /**
     * @param  string  $templateName
     * @param array $data
     *
     * @return void
     */
    private function options(string $templateName, array $data = []): void
    {
        $key = $this->telegramConfig['telegram_key'];
        if (empty($key)) {
            return;
        }

        // Initialize the messenger with logging.
        $this->messenger = new Telegram($key, $this->telegramConfig['timeout']);
        $this->messenger->setLog(new Log('telegram.log'), $this->telegramConfig['logs']);

        // Process user data and group data.
        $this->processData($this->telegramConfig['telegram_user_data'], $templateName, $data);
        $this->processData($this->telegramConfig['telegram_user_groupdata'], $templateName, $data);
    }

    /**
     * @param array|null $users
     * @param string $templateName
     * @param $data
     *
     * @return void
     */
    private function processData(?array $users, string $templateName, $data): void
    {
        if (empty($users)) {
            return;
        }

        foreach ($users as $user) {
            if (!empty($user['id_telegram']) && isset($user['checked']) && $user['checked'] === 'on') {
                if (!empty($this->uploaded_file)) {
                    $this->sendDocument($user, $templateName, $data, $this->uploaded_file);
                } else {
                    $this->sendMessage($user, $templateName, $data);
                }
            }
        }
    }


    /**
     * @param array $user
     * @param string $templateName
     * @param $data
     *
     * @return bool
     */
    private function sendMessage(array $user, string $templateName, $data): bool
    {
        $webhook = $this->telegramConfig['webhook'];
        $text = htmlspecialchars_decode($this->{$templateName . '_template'}($data));
        $this->messenger->setTo($user['id_telegram']);

        // Check if we can send directly via webhook
        if (!empty($webhook) && $webhook !== 'no' && $this->messenger->sendMessage($text, $this->telegramConfig['trim_messages'])) {
            return true;
        }

        // Attempt to send via proxy if enabled
        if ($this->telegramConfig['proxy'] == '1' && !empty($this->telegramConfig['telegram_proxy_data'])) {
            foreach ($this->telegramConfig['telegram_proxy_data'] as $proxy) {
                if (isset($proxy['status'], $proxy['ip'], $proxy['port'])) {
                    $proxyData = $proxy['ip'] . ':' . $proxy['port'];
                    if (!empty($proxy['login']) && !empty($proxy['password'])) {
                        $proxyData .= '@' . $proxy['login'] . ':' . $proxy['password'];
                    }

                    $this->messenger->setProxy($proxyData);

                    if ($this->messenger->sendMessage($text, $this->telegramConfig['trim_messages'])) {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    /**
     * @param array $user
     * @param string $templateName
     * @param $data
     *
     * @return bool
     */
    private function sendDocument(array $user, string $templateName, $data, $file): bool
    {
        $webhook = $this->telegramConfig['webhook'];
        $text = htmlspecialchars_decode($this->{$templateName . '_template'}($data));
        $this->messenger->setTo($user['id_telegram']);


        // Check if we can send directly via webhook
        if (!empty($webhook) && $webhook !== 'no' && $this->messenger->sendDocument($text, $file, $this->telegramConfig['trim_messages'])) {
            return true;
        }

        // Attempt to send via proxy if enabled
        if ($this->telegramConfig['proxy'] == '1' && !empty($this->telegramConfig['telegram_proxy_data'])) {

            foreach ($this->telegramConfig['telegram_proxy_data'] as $proxy) {
                if (isset($proxy['status'], $proxy['ip'], $proxy['port'])) {
                    $proxyData = $proxy['ip'] . ':' . $proxy['port'];
                    if (!empty($proxy['login']) && !empty($proxy['password'])) {
                        $proxyData .= '@' . $proxy['login'] . ':' . $proxy['password'];
                    }

                    $this->messenger->setProxy($proxyData);

                    if ($this->messenger->sendDocument($text, $file, $this->telegramConfig['trim_messages'])) {
                        return true;
                    }
                }
            }
        }

        return false;
    }



    /** ШАБЛОНЫ */

    private function call_me_template($data)
    {
        $message = '';
        if (!empty($this->telegramConfig['telegram_langdata'])) {
            $message = $this->telegramConfig['telegram_langdata']['call_me_telegram'];
        }

        $find = array(
            '{date_added}',
            '{fullName}',
            '{communication}',
            '{service}',
            '{phone}',
        );

        $feedback = $data[1];

        $replace = array(
            '{date_added}'    => date("Y-m-d H:i:s"),
            '{fullName}'      => $feedback['fullName'] ?? '',
            '{phone}'         => $feedback['phone'] ?? '',
            '{communication}' => $feedback['communication'] ?? '',
            '{service}'       => $feedback['service'] ?? '',

        );

        return str_replace(array("\r\n", "\r", "\n"), chr(10), preg_replace(array("/\s\s+/", "/\r\r+/", "/\n\n+/"), chr(10), str_replace($find, $replace, $message)));
    }

    private function feedback_template($data)
    {
        $message = '';
        if (!empty($this->telegramConfig['telegram_langdata'])) {
            $message = $this->telegramConfig['telegram_langdata']['feedback_telegram'];
        }

        $find = array(
            '{firstName}',
            '{lastName}',
            '{fullName}',
            '{phone}',
            '{email}',
            '{message}',
            '{communication}',
            '{service}',
            '{date_added}'
        );

        $feedback = $data[1];


        $replace = array(
            '{firstName}'     => $feedback['firstName'] ?? '',
            '{lastName}'      => $feedback['lastName'] ?? '',
            '{fullName}'      => $feedback['fullName'] ?? '',
            '{phone}'         => $feedback['phone'] ?? '',
            '{email}'         => $feedback['email'] ?? '',
            '{message}'       => $feedback['message'] ?? '',
            '{communication}' => $feedback['communication'] ?? '',
            '{service}'       => $feedback['service'] ?? '',
            '{date_added}'    => date("Y-m-d H:i:s")
        );

        return str_replace(array("\r\n", "\r", "\n"), chr(10), preg_replace(array("/\s\s+/", "/\r\r+/", "/\n\n+/"), chr(10), str_replace($find, $replace, $message)));
    }




}
