<?php

namespace Mnv\Core\Smarty;

use Mnv\Core\ConfigManager;
use Mnv\Core\Singleton\SingletonTrait;
use Smarty;
use SmartyException;
use RuntimeException;

/**
 * Class SmartyExtension
 * custom class Smarty
 *
 * @package Smarty
 */
final class Design extends Smarty
{

    use SingletonTrait;


    const PLUGINS_DIR = '/includes/smarty_plugins';
    const CONFIG_DIR = '/includes/smarty_config';

    /** @var bool */
    private $smartySecurity = false;

    /** @var string */
    private $defaultTemplateDir;

    /** @var array */
    private $smartyModifiers = [];

    /** @var array */
    private $smartyFunctions = [];

    /**
     * @var array
     */
    private $allowedPhpFunctions = [
        'escape',
        'cat',
        'count',
        'in_array',
        'nl2br',
        'str_replace',
        'reset',
        'floor',
        'round',
        'ceil',
        'max',
        'min',
        'number_format',
        'print_r',
        'var_dump',
        'printa',
        'file_exists',
        'stristr',
        'strtotime',
        'empty',
        'urlencode',
        'intval',
        'isset',
        'sizeof',
        'is_array',
        'array_intersect',
        'time',
        'array',
        'base64_encode',
        'implode',
        'explode',
        'preg_replace',
        'preg_match',
        'key',
        'json_encode',
        'json_decode',
        'is_file',
        'date',
        'strip_tags',
        'preg_quote'
    ];

    /**
     * Design constructor.
     * @throws SmartyException
     */
    public function __construct()
    {
        parent::__construct();

        /**
         * @var  $compile_check
         * При каждом вызове РНР-приложения Smarty проверяет, изменился или нет текущий шаблон с момента последней компиляции.
         * Если шаблон изменился, он перекомпилируется.
         * В случае, если шаблон еще не был скомпилирован, его компиляция производится с игнорированием значения этого параметра.
         * По умолчанию эта переменная установлена в true.
         * В момент, когда приложение начнет работать в реальных условиях (шаблоны больше не будут изменяться),
         * этап проверки компиляции становится ненужным.
         * В этом случае проверьте, чтобы переменная $compile_check была установлена в "false" для достижения максимальной производительности.
         * Учтите, что если вы присвоите этой переменной значение "false", и файл шаблона будет изменен, вы *НЕ* увидите изменений в выводе шаблона до тех пор,
         * пока шаблон не будет перекомпилирован.
         * Если caching и compile_check активированы, файлы кэша будут регенерированы при обновлении связанных с ним шаблонов или конфигурационных файлов
         */
        $this->compile_check   = Smarty::COMPILECHECK_ON;
        $this->error_reporting = E_ALL & ~E_NOTICE;

        $this->debugging = Smarty::DEBUG_OFF;

        $this->initializeSecurity();

        $this->initializeDirectories();
        $this->initializePlugins();

    }

    /**
     * @throws SmartyException
     */
    private function initializeSecurity(): void
    {
        if ($this->smartySecurity) {
            $this->enableSecurity();
            $this->security_policy->php_modifiers = $this->allowedPhpFunctions;
            $this->security_policy->php_functions = $this->allowedPhpFunctions;
            $this->security_policy->secure_dir = array(
                $this->defaultTemplateDir() . '/templates/',
                GLOBAL_ROOT . 'admin/templates',
                GLOBAL_ROOT . 'app',
            );
        }
    }

    /**
     * @return void
     */
    private function initializeDirectories(): void
    {
        $this->setConfigDir(GLOBAL_ROOT . self::CONFIG_DIR);
        $this->setCompileDir($this->tmpPath() . 'compile');
        $this->setCompileId('main' . SITE_LANG_POSTFIX . '-' . ConfigManager::getValue('theme'));
        $this->setTemplateDir($this->defaultTemplateDir());
        $this->setCacheDir($this->tmpPath() . 'cache');

        if (!is_dir($this->getCompileDir()) && !@mkdir($this->getCompileDir(), 0777, true)) {
            throw new RuntimeException("Unable to create directory: {$this->getCompileDir()}");
        }
    }

    /**
     * @return void
     * @throws SmartyException
     */
    private function initializePlugins(): void
    {
        $this->addPluginsDir(GLOBAL_ROOT . self::PLUGINS_DIR);
        $this->default_modifiers = ['escape'];
        SmartyPlugins::register($this);
    }

    /**
     * @return $this
     */
    public function design(): Design
    {
        return $this;
    }

    /**
     * @return string
     */
    public function tmpPath(): string
    {
        return sprintf('%s/temp/smarty/', GLOBAL_ROOT);
    }

    public function defaultTemplateDir(): string
    {
        return sprintf('%s/themes/' . ConfigManager::getValue('theme'), GLOBAL_ROOT);
    }

    public function defaultAdminTemplateDir(): string
    {
        return sprintf('%s/admin/templates', GLOBAL_ROOT);
    }


    /**
     * @param int $num
     * @param string|null $time
     */
    public function caching(int $num, ?string $time): void
    {
        $this->caching = $num;
        if (!empty($time)) {
            $this->cache_lifetime = $time;
        }
    }

    /** Установка директории файлов шаблона(отображения) */
    public function setTemplatesDir($dir): void
    {
        $dir = rtrim($dir, '/') . '/';
        if (!is_string($dir)) {
            throw new RuntimeException("Param \$dir must be string");
        }

        $this->defaultTemplateDir = $dir;
        $this->setSmartyTemplatesDir($dir);
    }

    public function setSmartyTemplatesDir($dir): void
    {
        $this->setTemplateDir($dir);
    }




}