<?php

namespace Mnv\Core\Table\Parser;

use Exception;
use Mnv\Core\Table\Exception\FileException;
/**
 * JSONParser для разбора HTML-таблицы
 *
 * @package JSONParser
 */
abstract class BaseParser implements IParser
{
    /**
     * Содержит проанализированный результат
     * @var      array
     */
    protected array $table;

    /**
     * Определяет допустимое расширение файла
     * @var string
     */
    protected string $file_extension = 'json';

    /**
     * @param string|null $file_url Path to file (optional)
     */
    public function __construct(?string $file_url = null)
    {
        if (isset($file_url)) {
            $this->loadFile($file_url);
        }
    }

    /**
     * Получить значение указанной ячейки
     *
     * @param int $row_num Row number
     * @param int $col Column number
     * @return array
     * @throws Exception If the cell identified doesn't exist.
     */
    public function getCell(int $row_num, int $col): array
    {
        // check whether the cell exists
        if (!$this->isCellExists($row_num, $col)) {
            throw new Exception("Cell $row_num,$col doesn't exist", FileException::CELL_NOT_FOUND);
        }
        return $this->table[$row_num][$col];
    }

    /**
     * Получить данные указанного столбца в виде массива
     *
     * @param int $col Column number
     * @return array
     * @throws Exception If the column requested doesn't exist.
     */
    public function getColumn(int $col): array
    {
        if (!$this->isColumnExists($col)) {
            throw new Exception("Column $col doesn't exist", FileException::COLUMN_NOT_FOUND);
        }

        return array_column($this->table, $col);
    }

    /**
     * Получить данные всех ячеек в виде массива
     *
     * @return array
     * @throws Exception If the field is not set.
     */
    public function getField(): ?array
    {
        return $this->isFieldExists() ? $this->table : null;
    }

    /**
     * Получить данные указанной строки в виде массива
     *
     * @param int $row_num Row number
     * @return array
     * @throws Exception When a row is requested that doesn't exist.
     */
    public function getRow(int $row_num): array
    {
        if (!$this->isRowExists($row_num)) {
            throw new Exception("Row $row_num doesn't exist", FileException::ROW_NOT_FOUND);
        }
        return $this->table[$row_num];
    }

    /**
     * Проверить, существует ли ячейка с указанной строкой и столбцом
     *
     * @param int $row_num Row number
     * @param int $col Column
     * @return bool
     */
    public function isCellExists(int $row_num, int $col): bool
    {
        return $this->isRowExists($row_num) && $this->isColumnExists($col);
    }

    /**
     * Проверить, существует ли указанный столбец
     *
     * @param int $col Column
     * @return bool
     */
    public function isColumnExists(int $col): bool
    {
        return isset($this->table[0][$col]);
    }

    /**
     * Проверить, существует ли указанная строка
     *
     * @param int $row_num Row number
     * @return bool
     */
    public function isRowExists(int $row_num): bool
    {
        return isset($this->table[$row_num]);
    }

    /**
     * Проверить, существует ли таблица
     *
     * @return bool
     */
    public function isFieldExists(): bool
    {
        return !empty($this->table);
    }

    /**
     * Проверьте, существует ли файл, действителен и доступен для чтения
     *
     * @param string $file_path Path to file
     * @return bool
     * @throws Exception If file being loaded doesn't exist
     * @throws Exception If file extension doesn't match
     * @throws Exception If error reading the file
     */
    public function isFileReady(string $file_path): bool
    {

        try {
            if (!file_exists($file_path)) {
                throw new FileException('File '.$file_path.' doesn\'t exist', FileException::FILE_NOT_FOUND);
            }

            $extension = strtoupper(pathinfo($file_path, PATHINFO_EXTENSION));
            if ($extension !== strtoupper($this->file_extension)) {
                throw new FileException('File extension mismatch: '.$extension, FileException::FILE_EXTENSION_MISMATCH);
            }

            $handle = fopen($file_path, 'r');
            if ($handle === false) {
                throw new FileException('Error reading file: '.$file_path, FileException::ERROR_READING_FILE);
            }

            return true;
        } finally {
            if (isset($handle) && is_resource($handle)) {
                fclose($handle);
            }
        }
    }
}
