<?php

namespace Mnv\Core\Table;

use Exception;
use Mnv\Core\Table\Parser\JSONParser;
use Mnv\Core\Table\Writer\JSONWriter;
use Mnv\Core\Table\Exception\FileException;

/**
 * Class JSONParser
 * @package JSONParser
 */
class TableFactory
{
    /** @var JSONParser */
    public JSONParser $parser;

    /** @var JSONWriter */
    public JSONWriter $writer;

    /**
     * Valid parser and writer types
     *
     * @var array
     */
    protected array $validTypes = ['XML', 'CSV', 'TSV', 'HTML', 'JSON'];

    /**
     * Cached instances for better performance
     *
     * @var array
     */
    protected static array $componentCache = [];

    /**
     * JSONParser constructor.
     * @param string $filetype Set the filetype of the file which will be parsed (XML/CSV/TSV/HTML/JSON)
     * @throws Exception
     */
    public function __construct(string $filetype = 'JSON')
    {
        $this->setFiletype(strtoupper($filetype));
    }

    /**
     * Set the filetype and initialize components lazily
     *
     * @param string $filetype Filetype for parser/writer
     * @throws Exception If filetype is not supported
     */
    protected function setFiletype(string $filetype): void
    {
        if (!in_array($filetype, $this->validTypes)) {
            throw new Exception("Unsupported filetype: '$filetype'", FileException::FILETYPE_NOT_SUPPORTED);
        }

        if (!isset(self::$componentCache[$filetype])) {
            self::$componentCache[$filetype] = [
                'Parser' => $this->createComponent($filetype, 'Parser'),
                'Writer' => $this->createComponent($filetype, 'Writer')
            ];
        }

        $this->parser = self::$componentCache[$filetype]['Parser'];
        $this->writer = self::$componentCache[$filetype]['Writer'];
    }

    /**
     * Create a parser or writer component.
     *
     * @param string $filetype  The filetype for the component (e.g., XML, CSV, etc.)
     * @param string $component Type of component ('Parser' or 'Writer')
     * @return object The created parser or writer object
     * @throws Exception If the filetype or class is unsupported
     */
    protected function createComponent(string $filetype, string $component): object
    {
        $class = "Mnv\Core\Table\\$component\\{$filetype}$component";
        if (!class_exists($class)) {
            throw new Exception("$component class '$class' not found.", FileException::CLASS_NOT_FOUND);
        }

        return new $class();
    }

    /**
     * Convert the current data to another file format.
     *
     * @param string $filetype The target file format (e.g., XML, CSV, etc.)
     * @throws Exception If the filetype is unsupported
     */
    public function convertTo(string $filetype = 'JSON'): void
    {
        $filetype = strtoupper($filetype);

        // Reuse components if already cached
        if (!isset(self::$componentCache[$filetype]['Writer'])) {
            self::$componentCache[$filetype]['Writer'] = $this->createComponent($filetype, 'Writer');
        }

        $this->writer = self::$componentCache[$filetype]['Writer'];
        $this->writer->setData($this->parser->getField());
    }


}