<?php

declare(strict_types=1);

namespace Mnv\Core\Table\Writer;

/**
 *Базовый класс JSONParser для написания электронной таблицы
 *
 * @package JSONParser
 */
abstract class BaseWriter implements IWriter
{
    /**
     * Хранит табличные данные
     *
     * @access  protected
     * @var     array
     */
    protected array $table = [];

    /**
     * Определяет тип содержимого для HTTP-заголовка
     *
     * @access  protected
     * @var     string
     */
    protected string $content_type = 'application/json';

    /**
     * Определяет расширение файла, которое будет использоваться при сохранении файла
     *
     * @access  protected
     * @var     string
     */
    protected string $file_extension = 'json';

    /**
     * @return  void
     */
    public function __construct()
    {
        $this->table = array();
    }

    /**
     * Добавление данных строки в таблицу
     *
     * @param   array   $values Массив содержит упорядоченное значение для каждой ячейки
     * @param   bool  $end  Проверьте, находится ли строка в начале или в конце массива
     * @return  void
     */
    public function addRow($values, bool $end = true)
    {
        $values = is_array($values) ? $values : [$values];
        $end ? $this->table[] = $values : array_unshift($this->table, $values);
    }

    /**
     *  Редактирование данных строки в таблицу
     *
     * @param int $index
     * @param array $values
     * @return bool
     */
    public function editRow(int $index, array $values): bool
    {
        if (!$this->isRowExists($index)) {
            return false;
        }

        $this->table[$index] = $values;
        return true;
    }

    /**
     * Удаление строки из таблицы
     *
     * @param int $index Индекс строки
     * @return bool Возвращает true, если строка успешно удалена, иначе false
     */
    public function deleteRow(int $index): bool
    {
        if (!$this->isRowExists($index)) {
            return false;
        }

        unset($this->table[$index]);
        // Пересобираем массив, чтобы избежать "дыр" в индексах
        $this->table = array_values($this->table);
        return true;
    }

    /**
     * Установить табличные данные
     *
     * @param   array   $values Массив содержит упорядоченное значение массивов для всех полей
     * @return  void
     */
    public function setData(array $values)
    {
        $this->table = $values;
    }


    /**
     * Проверка существования строки по индексу
     *
     * @param int $index Индекс строки
     * @return bool Возвращает true, если строка существует, иначе false
     */
    public function isRowExists(int $index): bool
    {
        return array_key_exists($index, $this->table);
    }


    /**
     * Сохранение данных таблицы в файл или вывод в браузер
     *
     * @param string      $filename Имя файла для сохранения (без расширения)
     * @param string|null $target   Путь сохранения файла или вывод в браузер
     * @return void
     */
    public function saveFile(string $filename, ?string $target = null): void
    {
        $filename = $filename ?: date('YmdHis');
        $target = $target ?: 'php://output';

        // Установить заголовки, если выводим в браузер
        if ($target === 'php://output') {
            header('Content-Type: ' . $this->content_type);
            header('Content-Disposition: attachment; filename=' . $filename . '.' . $this->file_extension);
        }

        file_put_contents($target, $this->saveString());

        // Если выводим в браузер, завершить выполнение скрипта
        if ($target === 'php://output') {
            exit();
        }
    }


//    /**
//     * Экспорт документа
//     *
//     * @param   string  $filename   Name for the saved file (extension will be set automatically)
//     * @param   string  $target     Save location
//     * @return  void
//     */
//    public function saveFile(string $filename, ?string $target = null)
//    {
//        if (!isset($filename)) {
//            $filename = date('YmdHis');
//        }
//
//        if (!isset($target)) {
//            // Написать выход на браузер
//            $target = 'php://output';
//
//            // Установить заголовок HTTP ответ
//            header('Content-Type: ' . $this->content_type);
//            header('Content-Disposition: attachment; filename=' . $filename . '.' . $this->file_extension);
//        }
//
//        $fp = fopen($target, 'w');
//        fwrite($fp, $this->saveString());
//        fclose($fp);
//
//        if ($target == 'php://output') {
//            // Так как не должно быть данных ниже
//            exit();
//        }
//    }

    /**
     * Преобразование данных таблицы в строку
     *
     * @return string Данные таблицы в виде строки
     */
    abstract protected function saveString(): string;


}