<?php

namespace Mnv\Core\Uploads;

use Mnv\Core\ConfigManager;
use Imagine\Image\Box;
use Imagine\Image\ImageInterface;
use Mnv\Core\Uploads\Exceptions\InvalidParamException;
use Symfony\Component\HttpFoundation\File\UploadedFile as SymfonyUploadedFile;

/**
 * Class ImagineGenerator
 * @package Mnv\Core\Uploads
 */
class ImagineGenerator extends AbstractGenerator implements IGenerator
{

    /** @var ImageInterface $image */
    private ImageInterface $image;

    /**
     * Конструктор класса.
     *
     * @param string      $realPath  Реальный путь до файла.
     * @param string      $path      Путь для сохранения файла.
     * @param string|null $uuid      Уникальный идентификатор файла (UUID).
     * @param int         $managerId ID менеджера, создавшего файл.
     */
    public function __construct(string $realPath, string $path, ?string $uuid, int $managerId)
    {
        // Вызываем конструктор родительского класса
        parent::__construct($realPath, $path, $uuid, $managerId);

        // Установка драйвера
        $this->setDriver();
    }

    /**
     * Инициализирует генератор с загруженным файлом.
     *
     * @param SymfonyUploadedFile $file Загруженный файл.
     * @return ImagineGenerator Текущий экземпляр генератора.
     * @throws InvalidParamException Если файл не может быть обработан.
     */
    public function init(SymfonyUploadedFile $file): ImagineGenerator
    {
        // Проверка валидности файла
        if (!$file->isValid() || !$file->getPathname() || !is_readable($file->getPathname())) {
            throw new InvalidParamException('The uploaded file is invalid or cannot be read.');
        }

        $this->file = $file;

        try {
            // Установка расширения файла
            $this->setGeneratorExtension($file->getClientOriginalExtension());

            // Генерация нового имени файла
            $this->generateFileName($file->getClientOriginalName());

            // Сбор опций для обработки шрифтов и водяных знаков
            $this->collectFontOptions();

            // Объединение опций обработки изображения
            $this->options = $this->mergeImageOptions();

            // Открытие изображения для обработки
            $this->image = $this->openImage($this->file->getPathname());
        } catch (\Throwable $e) {
            // Логируем ошибку
            error_log('Failed to initialize ImagineGenerator: ' . $e->getMessage());
            throw new InvalidParamException('Initialization of ImagineGenerator failed.', 0, $e);
        }

        return $this;
    }

    /**
     * Сохраняет изображение и его миниатюры.
     *
     * @return bool|int|string|null Результат сохранения файла (идентификатор записи или false).
     * @throws InvalidParamException Если возникли ошибки параметров.
     */
    public function save()
    {
        // Формируем массив данных изображения
        $imageData = $this->prepareImageData();

        // Сохраняем основное изображение
        $this->resizeAndSaveImage(
            $this->image->getSize()->getWidth(),
            $this->image->getSize()->getHeight(),
            $this->getFullImagePath()
        );

        // Обрабатываем и сохраняем миниатюры
        $this->processThumbnails();

        // Сохраняем информацию о файле
        return $this->saveFile($imageData);
    }

    /**
     * Подготавливает данные изображения для записи в базу или файл.
     *
     * @return array Массив с информацией о файле.
     */
    private function prepareImageData(): array
    {
        return [
            'name'      => "{$this->fileName}.{$this->extension}",
            'size'      => $this->file->getSize() ?? null,
            'mimeType'  => $this->getMimeType(),
        ];
    }

    /**
     * Определяет полный путь для сохранения основного изображения.
     *
     * @return string Полный путь к файлу.
     * @throws InvalidParamException
     */
    private function getFullImagePath(): string
    {
        // Проверяем, что `realPath` существует
        if (!is_dir($this->realPath)) {
            throw new InvalidParamException("Invalid real path: {$this->realPath} is not a directory.");
        }

        return rtrim($this->realPath, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . "{$this->fileName}.{$this->extension}";
    }

    /**
     * Определяет MIME-тип текущего файла.
     *
     * @return string MIME-тип файла.
     */
    private function getMimeType(): string
    {
        return $this->extension === 'webp' ? 'image/webp' : $this->file->getClientMimeType();
    }

    /**
     * Обрабатывает и сохраняет миниатюры для текущего изображения.
     *
     * @return void
     * @throws InvalidParamException
     */
    private function processThumbnails(): void
    {
        // Проверка доступности размеров
        if (empty($this->sizes)) {
            throw new InvalidParamException('The "sizes" property must be a non-empty array.');
        }

        foreach ($this->sizes as $size) {
            try {
                $this->resizeAndSaveThumbnail($size);
            } catch (InvalidParamException $e) {
                // Логируем ошибки для каждой миниатюры
                error_log("Error processing thumbnail for size '{$size}': " . $e->getMessage());
            }
        }
    }

    /**
     * Выполняет изменение размера изображения и его сохранение с учетом оптимизаций.
     *
     * @param int $width Ширина итогового изображения.
     * @param int $height Высота итогового изображения.
     * @param string $path Путь для сохранения файла.
     * @throws InvalidParamException
     */
    private function resizeAndSaveImage(int $width, int $height, string $path): void
    {
        // Проверка на существование временного файла
        if (!is_readable($this->file->getPathname())) {
            throw new InvalidParamException("Temporary file is not readable or does not exist: {$this->file->getPathname()}");
        }

        // Определяем изображение с водяным знаком или без него
        $image = $this->shouldApplyWatermark()
            ? $this->applyWatermark()
            : $this->resizeImage($width, $height);

        // Сохраняем изображение в указанное место
        $this->saveImage($image, $width, $height, $path);
    }

    /**
     * Проверяет необходимость применения водяного знака.
     *
     * @return bool
     */
    private function shouldApplyWatermark(): bool
    {
        return ConfigManager::getValue('allow_watermark') && $this->extension !== 'gif';
    }

    /**
     * Применяет водяной знак.
     *
     * @return ImageInterface
     * @throws InvalidParamException
     */
    private function applyWatermark(): ImageInterface
    {
        return BaseImage::textBox($this->file->getPathname(), ConfigManager::getValue('watermark_text'), $this->fontWatermark, ConfigManager::getValue('watermark_position'), $this->fontOptions);
    }

    /**
     * Изменяет размер изображения без водяного знака.
     *
     * @param int $width
     * @param int $height
     * @return ImageInterface
     * @throws InvalidParamException
     */
    private function resizeImage(int $width, int $height): ImageInterface
    {
        return BaseImage::resize($this->file->getPathname(), $width, $height);
    }

    /**
     * Сохраняет изображение с параметрами.
     *
     * @param ImageInterface $image Инстанс обработанного изображения.
     * @param int $width Ширина итогового изображения.
     * @param int $height Высота итогового изображения.
     * @param string $path Путь для сохранения файла.
     * @throws InvalidParamException
     */
    private function saveImage(ImageInterface $image, int $width, int $height, string $path): void
    {
        try {
            $image->thumbnail(new Box($width, $height))->save($path, $this->options);
        } catch (\Exception $e) {
            error_log("Error saving image to '{$path}': " . $e->getMessage());
            throw new InvalidParamException("Failed to save image: {$e->getMessage()}", 0, $e);
        }
    }

    /**
     * Создает и сохраняет миниатюру изображения заданного размера.
     *
     * @param string $size Размер миниатюры (например, 'small', 'medium', 'large').
     * @return void
     * @throws InvalidParamException Если размер миниатюры не указан или некорректен.
     */
    private function resizeAndSaveThumbnail(string $size): void
    {
        // Получение настроек размера миниатюры из конфигурации
        $thumbConfigKey = "{$size}_thumb";
        $thumbnailSizes = $this->autoSize(ConfigManager::getValue($thumbConfigKey));

        if (empty($thumbnailSizes) || count($thumbnailSizes) !== 2) {
            throw new InvalidParamException("Invalid thumbnail configuration for key: {$thumbConfigKey}");
        }

        [$width, $height] = $thumbnailSizes;

        // Формируем путь для сохранения миниатюры
        $thumbnailPath = rtrim($this->realPath, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $size . DIRECTORY_SEPARATOR . $this->fileName . '.' . $this->extension;

        // Сохраняем изображение
        $this->resizeAndSaveImage($width, $height, $thumbnailPath);
    }

    /**
     * Объединяет параметры изображения для обработки.
     *
     * @return array Массив параметров для обработки изображения.
     */
    private function mergeImageOptions(): array
    {
        try {
            return BaseImage::mergeOptions($this->extension, $this->thumbnailQuality);
        } catch (InvalidParamException $e) {
            // Логируем ошибку (исключая вывод на экран в продакшене)
            error_log("Failed to merge image options: " . $e->getMessage());

            // Возвращаем параметры по умолчанию
            return [
                'quality' => $this->thumbnailQuality, // Здесь задается уровень качества по умолчанию
            ];
        }
    }

    /**
     * Открывает изображение по указанному пути.
     *
     * @param string $path Путь к изображению.
     * @return ImageInterface Инстанс обработанного изображения.
     * @throws InvalidParamException Если изображение невозможно открыть.
     */
    private function openImage(string $path): ImageInterface
    {
        try {
            return BaseImage::open($path);
        } catch (InvalidParamException $e) {
            // Логируем подробности ошибки
            error_log("Error opening image at path '{$path}': " . $e->getMessage());
            // Повторный выброс исключения для дальнейшей обработки
            throw $e;
        }
    }

    /**
     * Устанавливает драйвер обработки изображений.
     *
     * @throws \RuntimeException Если драйвер не поддерживается.
     */
    private function setDriver(): void
    {
        // Доступные драйверы
        $drivers = [
            'imagick'  => \Imagine\Imagick\Imagine::class,
            'gd'       => \Imagine\Gd\Imagine::class,
            'gmagick'  => \Imagine\Gmagick\Imagine::class,
        ];

        // Определяем драйвер по умолчанию: Imagick, если доступен
        if ($this->imageLibrary != 2) {
            if (extension_loaded('imagick') && class_exists('Imagick')) {
                $this->driver = 'imagick';

                // Проверка поддержки WebP. Если не поддерживается, переключаемся на GD (если доступен)
                if (!$this->supportsWebPWithImagick() && function_exists('imagewebp') && $this->imageLibrary != 1) {
                    $this->driver = 'gd';
                }
            }
        }

        // Устанавливаем драйвер, если он есть в списке поддерживаемых
        if (isset($drivers[$this->driver])) {
            BaseImage::setImagine(new $drivers[$this->driver]());
        } else {
            throw new \RuntimeException("Unsupported image driver: {$this->driver}");
        }
    }

    /**
     * Проверяет, поддерживает ли Imagick формат WebP.
     *
     * @return bool
     */
    private function supportsWebPWithImagick(): bool
    {
        return extension_loaded('imagick') && \Imagick::queryFormats('WEBP') !== [];
    }


}