<?php

namespace Mnv\Models;

use Mnv\Core\Model;
use Mnv\Core\SortTable;
use Mnv\Core\Test\Logger;
use Mnv\Core\Utilities\Cookie\Session;
use Mnv\Http\Request;
use Mnv\Models\Exceptions\NoContentException;
use Mnv\Models\Exceptions\NotInsertContentException;
use Mnv\Models\Exceptions\NotUpdateContentException;
use Mnv\Models\Exceptions\NotContentApproveException;


/**
 * Class Content
 * @package Mnv\Models\Contents
 */
class Content extends Model
{
    /** @var string */
    protected string $table = 'articles';

    /** @var string  */
    protected string $table_image = 'article_images';

    /** @var string  */
    protected string $primaryKey = 'articleId';

    /** @var string  */
    protected string $orderBy = 'publishedOn DESC';

    /** @var string  */
    protected string $columns = '*';

    /** @var string  */
    protected string $table_comments = 'comments';

    /** @var string  */
    protected string $table_types_content = 'types_content';

    /** @var string  */
    protected string $table_custom_fields = 'type_content_custom_fields';

    /** @var string  */
    protected string $table_custom_values = 'type_content_custom_values';

    public $field;

    /** @var array */
    public $image = [];

    /** @var int $imageId */
    public int $imageId;

    /** @var string  */
    public string $contentType;

    /** @var mixed  */
    public $group_action;

    /** @var mixed  */
    public $contentIds;

    public function __construct(Request $request)
    {
        $this->id           = $request->get('id');
        $this->data         = $request->get('article');
        $this->field        = $request->get('field');
        $this->contentIds   = $request->get('articleIds');
        $this->group_action = $request->get('group_action');

        $this->sortBy       = $request->get('sortBy');
        $this->sortOrder    = $request->get('sortOrder');
    }


    /** Фильтрация / сортировка */
    protected function sorting(): void
    {
        global $SECTIONS;

        if (!empty($this->filter['contentType'])) {
            connect()->where('contentType', $this->filter['contentType']);
        }

        if (!empty($this->filter['status'])) {
            connect()->where('status', $this->filter['status']);
        }


        if (!empty($this->filter['query'])) {
//            connect()->grouped(function($q) {
//                $q->like('title', "%" . $this->filter['query'] . "%")->orLike('content', "%" . $this->filter['query'] . "%")->orLike('keywords',"%" . $this->filter['query'] . "%");
//            });

            connect()->whereAny(['title', 'content', 'keywords'], 'like', "%" . $this->filter['query'] . "%");
        }

        if (!empty($this->filter['section']) && !empty($SECTIONS[$this->filter['section']])) {
            $sectionIds = array($this->filter['section']);
            if (!empty($SECTIONS[$this->filter['section']]['allChildren'])) {
                $sectionIds = array_merge($sectionIds, $SECTIONS[$this->filter['section']]['allChildren']);
            }
            connect()->in('sectionId', $sectionIds);
        }

        if (!empty($this->filter['dateStart']) && !empty($this->filter['dateEnd'])) {
            connect()->between('publishedOn', $this->filter['dateStart'], $this->filter['dateEnd']);
        }

    }

    /** Получение всех статей */
    public function all($limit, $page)
    {
        /** Фильтрация / сортировка */
        $this->sorting();

        return parent::all($limit, $page);
    }

    /** Получение кол-во контента */
    public function total(): void
    {
        /** Фильтрация / сортировка */
        $this->sorting();
        parent::total();
    }

    /**
     * Проверка на совпадение и получение fileName
     *
     * @param string|null $fileName
     * @return int|mixed|string|null
     */
    public function checkFileName(?string $fileName)
    {
        if (empty($fileName)) {
            $maxId = $this->getMaxValue($this->primaryKey);
            return $maxId ? $maxId + 1 : 1;
        }

        if (!empty($this->id)) connect()->where($this->primaryKey,'<>', $this->id);
        if ($fileName = connect()->table($this->table)->select('fileName')->where('LOWER(fileName)', strtolower($fileName))->getValue()) {
            return $fileName;
        }

        return null;
    }

    /**
     * @throws NotUpdateContentException
     * @throws NotInsertContentException
     */
    public function prepare(array $data, int $managerId): bool
    {

        $data['modifiedBy']   = $managerId;
        $data['modifiedOn']   = gmdate('Y-m-d H:i:s');
        $data['publishedOn']  = adjustTime(date('Y-m-d H:i:s', strtotime($data['publishedOn'])),  true);

        if (empty($this->id)) {

            $data['addedBy'] = $managerId;
            $data['addedOn'] = gmdate('Y-m-d H:i:s');
            $data['orderBy'] = connect($this->table)->max('orderBy')->where('contentType', $data['contentType'])->getValue() + 1;
            if ($this->id = $this->insert($data)) {
                $this->setCustomFields();
                Logger::init()->info("Добавлен новый контент «" . $data['title'] . "»", $managerId)->save();
                return true;
            }
            Logger::init()->error("Ошибка при добавление контента «" . $data['name'] . "»", $managerId)->save();

            throw new NotInsertContentException();
        }
        else {

            if ($this->update($data)) {
                $this->setCustomFields();
                Logger::init()->info("В контент «" . $data['title'] . "» были внесены изменения", $managerId)->save();
                return true;
            }
            Logger::init()->error("Ошибка при редактирование контента «" . $data['title'] . "»", $managerId)->save();

            throw new NotUpdateContentException();
        }

    }


    /**
     * Удаление контента
     *
     * @return bool
     *
     * @throws Exceptions\NotFoundException
     * @throws NoContentException
     * @throws \Mnv\Core\Database\Throwable\DatabaseException
     */
    public function remove(): bool
    {
        if (parent::remove()) {
            connect($this->table_comments)->where($this->primaryKey, $this->id)->delete();
            connect($this->table_custom_values)->where($this->primaryKey, $this->id)->delete();

            return true;
        }

        return false;
    }


    /**
     * Сортировка контента
     *
     * @param $ids
     * @param $articleIds
     */
    public function reorder($ids, $articleIds): void
    {
        foreach ($articleIds as $i => $articleId) {
            connect()->table($this->table)->where($this->primaryKey, $ids[$i])->update(['orderBy' => $articleId]);
        }
    }

    /**
     * Групповые действия
     */
    public function group(): void
    {
        if ($this->group_action === 'status') {
            if ($this->data = $this->get()) {
                $contentUpdate['status'] = ($this->data['status'] === 'V') ? 'H' : 'V';
                $this->statusContent($contentUpdate);
            }

        } else if ($this->group_action === 'remove') {
            if ($this->data = $this->get()) {
                $this->removeContent();
            }
        }
    }

    /**
     * Обновление статуса контента
     *
     * @param $contentUpdate
     * @return bool
     */
    public function statusContent($contentUpdate): bool
    {
        if (connect()->table($this->table)->where($this->primaryKey, $this->id)->update($contentUpdate)) {
            return true;
        }

        return false;
    }

    /** Групповое удаление контента */
    public function removeContent(): bool
    {
        /** удаление контента и удаление записей из базы данных прикрепленных картинок и комментариев к этому контенту */
        if (connect()->table($this->table)->where($this->primaryKey, $this->id)->delete()) {
            connect()->table($this->table_comments)->where($this->primaryKey, $this->id)->delete();
            connect()->table($this->table_image)->where($this->primaryKey, $this->id)->delete();

            return true;
        }

        return false;
    }


    /** **************** CUSTOM FIELDS *************** */

    /**
     * Обрабатывает и обновляет кастомные поля для текущего типа контента.
     *
     * Этот метод получает активные кастомные поля, проверяет каждое поле и
     * обновляет или удаляет их в зависимости от значений.
     *
     * @return void
     */
    public function setCustomFields()
    {
        // Проверяем, что поле 'field' является массивом
        if (!is_array($this->field)) {
            return;
        }

        // Получаем активные кастомные поля для текущего типа контента
        $activeCustomFields = $this->getActiveCustomFields();

        // Если есть активные кастомные поля
        if (!empty($activeCustomFields)) {
            // Массивы для полей, которые нужно обновить и удалить
            $fieldsToUpdate = [];
            $fieldsToDelete = [];

            // Проходим по всем полям и разделяем их на обновляемые и удаляемые
            foreach ($this->field as $fileName => $fieldValue) {
                if (in_array($fileName, $activeCustomFields)) {
                    // Подготовка данных для обновления, если поле активно
                    $fieldsToUpdate[] = [
                        'articleId' => $this->id,
                        'fileName'  => $fileName,
                        'value'     => $fieldValue
                    ];
                } elseif ($fieldValue === null) {
                    // Подготовка данных для удаления, если значение поля пусто
                    $fieldsToDelete[] = $fileName;
                }
            }

            // Массовое обновление кастомных полей
            if (!empty($fieldsToUpdate)) {
                $this->updateCustomFieldsBatch($fieldsToUpdate);
            }

            // Массовое удаление кастомных полей с пустыми значениями
            if (!empty($fieldsToDelete)) {
                $this->deleteCustomFieldsBatch($this->id, $fieldsToDelete);
            }
        }
    }

    /**
     * Получает список активных кастомных полей для текущего типа контента.
     *
     * @return array Список имен файлов активных кастомных полей.
     */
    private function getActiveCustomFields(): array
    {
        return connect($this->table_custom_fields . ' AS cf')
            ->usingJoin($this->table_types_content . ' AS tc', 'typeId')
            ->select('cf.fileName')
            ->where('cf.status', 'V') // Статус поля 'V' (активный)
            ->where('tc.fileName', $this->contentType)
            ->pluck('fileName');
    }

    /**
     * Массовое обновление кастомных полей.
     *
     * @param array $fieldsToUpdate Массив данных для обновления кастомных полей.
     * @return void
     */
    private function updateCustomFieldsBatch(array $fieldsToUpdate): void
    {
        // Выполняем массовое обновление или вставку кастомных полей
        foreach ($fieldsToUpdate as $fieldData) {
            connect()->table($this->table_custom_values)->replace($fieldData);
        }
    }

    /**
     * Массовое удаление кастомных полей с пустыми значениями.
     *
     * @param int $contentId ID записи контента.
     * @param array $fieldsToDelete Список имен файлов кастомных полей для удаления.
     * @return void
     */
    private function deleteCustomFieldsBatch(int $contentId, array $fieldsToDelete): void
    {
        connect()->table($this->table_custom_values)->where($this->primaryKey, $contentId)->whereIn('fileName', $fieldsToDelete)->delete();
    }


    public function getCustomFieldsValues($articleId): array
    {
        $result = connect($this->table_custom_values)
            ->usingJoin($this->table_custom_fields , 'fileName')
            ->select('fileName, value')
            ->where($this->primaryKey, $articleId)
            ->where('status', 'V')
            ->orderBy('orderBy')
            ->pluck('value', 'fileName');

//        print_r(connect()->getQuery());
        if (!empty($result)) {
            return $result;
        }

        return [];
    }


}