<?php

namespace Mnv\Models;

use Mnv\Core\Model;
use Mnv\Http\Request;
use Mnv\Models\Exceptions\NoContentException;
use Mnv\Models\Exceptions\NotFoundException;

/**
 * Class ContentType
 * @package Mnv\Models
 */
class ContentType extends Model
{

    /** @var string */
    protected string $table = 'types_content';
    protected string $table_custom_fields = 'type_content_custom_fields';
    protected string $table_custom_values = 'type_content_custom_values';

    /** @var string  */
    protected string $primaryKey = 'typeId';

    /** @var string  */
    protected string $orderBy = 'orderBy ASC';


    /** @var array|mixed  */
    public $groupIds = [];

    /** @var array|mixed  */
    public $typeIds = [];

    public int $fieldId;

    /** @var array|mixed  */
    public $field = [];

    /** @var array|null  */
    public ?array $fields = [];

    /** @var array|null  */
    public ?array $modules = [];


    public function __construct(Request $request)
    {
        $this->id       = (int)$request->get('id');
        $this->data     = $request->get('type');
        $this->modules  = $request->get('modules');
        $this->groupIds = $request->get('groupIds');
        $this->typeIds  = $request->get('ids');
        $this->field    = $request->get('field');
    }

    /**
     * Проверка на совпадение и получение fileName
     *
     * @param string|null $fileName
     * @return mixed|string|null
     */
    public function checkFileName(?string $fileName)
    {
        if (!empty($this->id)) {
            connect()->where($this->primaryKey,'<>', $this->id);
        }

        if ($fileName = connect($this->table)->select('fileName')->where('LOWER(fileName)', strtolower($fileName))->getValue()) {
            return $fileName;
        }

        return null;
    }


    public function prepare(array $data, int $managerId): bool
    {

        $data['modifiedBy']   = $managerId;
        $data['modifiedOn']   = gmdate('Y-m-d H:i:s');

        if (empty($this->id)) {

            $data['addedBy']        = $managerId;
            $data['addedOn']        = gmdate('Y-m-d H:i:s');
            $data['orderBy']        = $this->getMaxValue('orderBy') + 1;

            if ($this->id = $this->insert($data)) {
                $this->privileges($data['fileName']);
                $this->generateMenu();
                return true;
            }
        } else {
            if ($this->update($data)) {
                $this->privileges($data['fileName']);
                $this->generateMenu();

                return  true;
            }
        }

        return false;
    }


    /**
     * Удаление
     *
     * @return bool
     *
     * @throws NoContentException
     * @throws NotFoundException
     */
    public function remove(): bool
    {
        if (empty($this->id)) {
            throw new NotFoundException("Not Found: ID is required.");
        }

        $currentRecord = connect($this->table)->where($this->primaryKey, $this->id)->get('array');
        // Throw exception if no record is found for the given `id`
        if (!$currentRecord) {
            throw new NoContentException("No Content: Record not found for the given ID.");
        }

        connect($this->table)->where($this->primaryKey, $this->id)->delete();
        $fields = connect($this->table_custom_fields)->where($this->primaryKey, $this->id)->getAll('array');
        if ($fields) {

            // удалить поля
            connect($this->table_custom_fields)->where($this->primaryKey, $this->id)->delete();

            foreach ($fields as $field) {
                // удалить value
                connect($this->table_custom_values)->where('fileName', $field['fileName'])->delete();
            }
        }

        // получаем контент
        $contentIds = connect('articles')->select('articleId')->where('contentType', $currentRecord['fileName'])->pluck('articleId', 'articleId');
        if ($contentIds) {
            // удалить контент
            connect('articles')->where('contentType', $currentRecord['fileName'])->delete();
            // удаляем прикрепленные картинки
            connect('article_images')->in('articleId', $contentIds)->delete();
        }

        $this->generateMenu();

        return true;

    }


    /**
     * Toggles the status of the current record between 'V' (Visible) and 'H' (Hidden).
     *
     * @return bool True if status was successfully toggled.
     *
     * @throws NotFoundException if `id` is missing or invalid.
     * @throws NoContentException if no record is found for the given `id`.
     */
    public function status(): bool
    {
        // Ensure that `id` is provided; throw exception if it's missing
        if (empty($this->id)) {
            throw new NotFoundException("Not Found: ID is required.");
        }

        // Retrieve the current `status` from the database based on `id`
        $currentRecord = connect($this->table)->select('status')->where($this->primaryKey, $this->id)->get('array');

        // Throw exception if no record is found for the given `id`
        if (!$currentRecord) {
            throw new NoContentException("No Content: Record not found for the given ID.");
        }
        // Toggle status between 'V' (Visible) and 'H' (Hidden)
        $this->status = ($currentRecord['status'] === 'V') ? 'H' : 'V';
        connect($this->table)->where($this->primaryKey, $this->id)->update(['status' => $this->status]);

        // Update internal status and regenerate the menu
        $this->generateMenu();

        return true;


    }

    /** USERS PRIVILEGES  */

    public function getPrivileges()
    {
        $this->data['privilege'] = connect('user_group_privileges')->where('privilege', $this->data['fileName'])->pluck('groupId', 'groupId');
    }

    /**
     * Добавление доступа к разделам меню
     * @param $privilege
     *
     * @return void
     */
    private function privileges($privilege): void
    {
        if (!empty($this->groupIds)) {
            connect('user_group_privileges')->where('privilege', $privilege)->delete();
            foreach ($this->groupIds as $groupId) {
                connect('user_group_privileges')->insert([
                    'groupId'   => $groupId,
                    'privilege' => $privilege
                ]);
            }

        }
    }

    private function generateMenu(): void
    {
        $url_menu_file = GLOBAL_ROOT . "/admin/menu/menu.json";

        $menu = json_decode(file_read($url_menu_file), true);
        $newItems = connect($this->table)->select('name, fileName')->orderBy('orderBy ASC')->where('status', 'V')->getAll('array');

        foreach ($menu as $key => $item) {

            if (in_array('contents', $item['activeItems'], true)) {
                $contents = array();
                foreach ($newItems as $newItem) {
                    if (in_array($newItem['fileName'], $item['children'])) {
                        continue;
                    } else {
                        $contents[] = [
                            'name'      => $newItem['name'],
                            'fileName'  => $newItem['fileName']
                        ];
                    }
                }

                $menu[$key]['children'] = $contents;

                $activeItems = array('contents');
                foreach($contents as $content) {
                    $activeItems[] = $content['fileName'];
                }

                $menu[$key]['activeItems'] = $activeItems;
            }
        }

        $json_menu = json_encode($menu, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT);
        if ($fh = fopen($url_menu_file, 'w')) {
            fwrite($fh, $json_menu);
            fclose($fh);

            $this->generatePermissions();
        }

    }

    public function generatePermissions(): bool
    {
        $modulesPermissions = include GLOBAL_ROOT.'/admin/config/modules.permissions.inc.php';
        $newItems = connect()->table($this->table)->select('name, fileName')->orderBy('orderBy ASC')->where('status', 'V')->getAll('array');
        $newPermissions = array();

        foreach ($modulesPermissions as $controller => $modulePermission)  {
            $newPermissions[$controller] = $modulePermission;
            if ($controller == 'ContentsAdmin') {
                $activeItems = array();
                foreach($newItems as $content) {
                    $activeItems[] = $content['fileName'];
                }
                $newPermissions[$controller] = $activeItems;
            }
            if ($controller == 'ContentAdmin') {
                $activeItems = array();
                foreach($newItems as $content) {
                    $activeItems[] = mb_substr($content['fileName'], 0, -1);
                }
                $newPermissions[$controller] = $activeItems;
            }
        }

        $permissionExport = "<?php\r\n";
        $permissionExport .= "return \$modules_permissions = ". var_export($newPermissions, true) . ";\r\n";
        $permissionExport .= "?>";

        if ( $fh = fopen(GLOBAL_ROOT.'/admin/config/modules.permissions.inc.php', 'w') ) {
            fwrite($fh, $permissionExport);
            fclose($fh);

            return true;
        }
        return false;
    }



    /** STATIC METHODS */


    /**
     * @param string $orderBy
     * @param string $indexKey
     * @param string $valueKey
     *
     * @return array|null
     */
    public static function selectContentTypes(string $orderBy, string $indexKey, string $valueKey): ?array
    {
        return connect('types_content')->orderBy($orderBy)->where('status', 'V')->pluck($valueKey, $indexKey);
    }

    /**
     * @param string $field
     * @param string $fileName
     *
     * @return string|null
     */
    public static function getContentTypeValue(string $field, string $fileName): ?string
    {
        return connect('types_content')->select($field)->where('fileName', $fileName)->getValue();
    }

    /**
     *
     * @param string $fileName
     *
     * @return array|null
     */
    public static function getContentType(string $fileName): ?array
    {
        $type = connect('types_content')->select('typeId, name, modules')->where('fileName', $fileName)->get('array');
        $type['modules'] = $type['modules'] ? json_decode($type['modules'], true) : [];

        return $type;
    }

}