<?php

namespace Mnv\Models\Form;

use Mnv\Core\Database\Throwable\DatabaseException;
use Mnv\Core\Model;
use Mnv\Http\Request;
use Mnv\Models\Exceptions\NoContentException;
use Mnv\Models\Exceptions\NotFoundException;

/**
 * Class Form
 * @package Mnv\Models
 */
class Form extends Model
{

    /** @var string */
    protected string $table = 'forms';

    /** @var string  */
    protected string $table_image = 'form_images';

    /** @var string  */
    protected string $table_fields = 'form_fields';

    /** @var string  */
    protected string $primaryKey = 'formId';

    /** @var string  */
    protected string $orderBy = 'orderBy ASC';


    /** @var array|mixed  */
    public $groupIds = [];

    /** @var array|mixed  */
    public $typeIds = [];

    public int $fieldId;

    /** @var array|mixed  */
    public $field = [];

    /** @var array|null  */
    public ?array $fields = [];

    public array $methods = [
        'POST' => 'POST',
        'GET'   => 'GET'
    ];


    public function __construct(Request $request)
    {
        $this->id       = (int)$request->get('id');
        $this->data     = $request->get('form');
        $this->groupIds = $request->get('groupIds');
        $this->typeIds  = $request->get('ids');
        $this->field    = $request->get('field');
    }

    /**
     * Проверка на совпадение и получение fileName
     *
     * @param string|null $fileName
     * @return mixed|string|null
     */
    public function checkFileName(?string $fileName)
    {
        if (!empty($this->id)) {
            connect()->where($this->primaryKey,'<>', $this->id);
        }

        if ($fileName = connect($this->table)->select('fileName')->where('LOWER(fileName)', strtolower($fileName))->getValue()) {
            return $fileName;
        }

        return null;
    }

    public function prepare(array $data, int $managerId): bool
    {

        $data['modifiedBy']   = $managerId;
        $data['modifiedOn']   = gmdate('Y-m-d H:i:s');

        if (empty($this->id)) {

            $data['addedBy']        = $managerId;
            $data['addedOn']        = gmdate('Y-m-d H:i:s');
            $data['orderBy']        = $this->getMaxValue('orderBy') + 1;

            if ($this->id = $this->insert($data)) {
                return true;
            }
        } else {
            if ($this->update($data)) {

                return  true;
            }
        }

        return false;
    }


    /**
     * Удаление
     *
     * @return bool
     *
     * @throws NoContentException
     * @throws NotFoundException
     * @throws DatabaseException
     */
    public function remove(): bool
    {
        if (empty($this->id)) {
            throw new NotFoundException("Not Found: ID is required.");
        }

        $currentRecord = connect($this->table)->where($this->primaryKey, $this->id)->get('array');

        // Throw exception if no record is found for the given `id`
        if (!$currentRecord) {
            throw new NoContentException("No Content: Record not found for the given ID.");
        }

        $deleted = connect($this->table)->where($this->primaryKey, $this->id)->delete();
        if (!$deleted) {
            throw new DatabaseException("Error: Failed to delete record with ID {$this->id}.");
        }

        // Delete from image table if it exists
        if (!empty($this->table_image)) {
            connect($this->table_image)->where($this->primaryKey, $this->id)->delete();
        }

        $fields = connect($this->table_fields)->where($this->primaryKey, $this->id)->getAll('array');
        if ($fields) {
            // удалить поля
            connect($this->table_fields)->where($this->primaryKey, $this->id)->delete();
        }

        return true;

    }


    /**
     * Toggles the status of the current record between 'V' (Visible) and 'H' (Hidden).
     *
     * @return bool True if status was successfully toggled.
     *
     * @throws NotFoundException if `id` is missing or invalid.
     * @throws NoContentException if no record is found for the given `id`.
     */
    public function status(): bool
    {
        // Ensure that `id` is provided; throw exception if it's missing
        if (empty($this->id)) {
            throw new NotFoundException("Not Found: ID is required.");
        }

        // Retrieve the current `status` from the database based on `id`
        $currentRecord = connect($this->table)->select('status')->where($this->primaryKey, $this->id)->get('array');

        // Throw exception if no record is found for the given `id`
        if (!$currentRecord) {
            throw new NoContentException("No Content: Record not found for the given ID.");
        }
        // Toggle status between 'V' (Visible) and 'H' (Hidden)
        $this->status = ($currentRecord['status'] === 'V') ? 'H' : 'V';
        connect($this->table)->where($this->primaryKey, $this->id)->update(['status' => $this->status]);

        return true;


    }



    /** STATIC METHODS */


    /**
     * @param string $orderBy
     * @param string $indexKey
     * @param string $valueKey
     *
     * @return array|null
     */
    public static function selectContentTypes(string $orderBy, string $indexKey, string $valueKey): ?array
    {
        return connect('types_content')->orderBy($orderBy)->where('status', 'V')->pluck($valueKey, $indexKey);
    }

    /**
     * @param string $field
     * @param string $fileName
     *
     * @return string|null
     */
    public static function getContentTypeValue(string $field, string $fileName): ?string
    {
        return connect('types_content')->select($field)->where('fileName', $fileName)->getValue();
    }

    /**
     *
     * @param string $fileName
     *
     * @return array|null
     */
    public static function getContentType(string $fileName): ?array
    {
        $type = connect('types_content')->select('typeId, name, modules')->where('fileName', $fileName)->get('array');
        $type['modules'] = $type['modules'] ? json_decode($type['modules'], true) : [];

        return $type;
    }


}