<?php

namespace Mnv\Models\Form;

use Mnv\Core\Model;
use Mnv\Http\Request;
use Mnv\Models\Exceptions\NoContentException;
use Mnv\Models\Exceptions\NotFoundException;

/**
 * Class FormField
 * @package Mnv\Models
 */
class FormField extends Model
{
    /** @var string */
    protected string $table = 'form_fields';

    /** @var string  */
    protected string $primaryKey = 'fieldId';

    /** @var string  */
    protected string $orderBy = 'orderBy ASC';

    /** @var int|null  */
    public $formId;

    /** @var  */
    public $properties;

    public $fieldIds = [];

    /** @var array|null  */
    public ?array $options = [];
    public ?array $attributes = [];

    /** @var array  */
    public static array $fieldTypes = [
        'I'   => 'INPUT',
        'T'   => 'TEXTAREA',
        'S'   => 'SELECT',
        'C'   => 'CHECKBOX',
        'R'   => 'RADIO',
        'B'   => 'BUTTON',
    ];

    public static array $inputTypes = [
        'text'      => 'TEXT',
        'email'     => 'EMAIL',
        'tel'       => 'TEL',
        'number'    => 'NUMBER',
        'hidden'    => 'HIDDEN',
        'file'      => 'FILE',
        'color'     => 'COLOR',
        'month'     => 'MONTH',
        'date'      => 'DATE',
        'time'      => 'TIME',

    ];

    public static array $buttonTypes = [
        'button'    => 'BUTTON',
        'submit'    => 'SUBMIT',
        'reset'     => 'RESET',

    ];

    public array $groupPositions = [
            '0' => 'Выберите расположение группы',
            'V' => 'Вертикально',
            'H' => 'Горизонтально'
    ];

    public function __construct(Request $request)
    {
        $this->id           = $request->get('id');
        $this->formId       = $request->get('formId');

        $this->data         = $request->get('field');
        $this->options      = $request->get('options');
        $this->attributes   = $request->get('attributes');


        $this->fieldIds     = $request->get('ids');

    }


    /**
     * Edit a record by ID
     *
     * @return $this
     */
    public function edit(): FormField
    {
        if (empty($this->id)) {
            return $this;
        }

        $this->data = $this->get();

        $this->options = !empty($this->data['options'])
            ? json_decode($this->data['options'], true)
            : [];


        $this->attributes = !empty($this->data['attributes'])
            ? json_decode($this->data['attributes'], true)
            : [];


        return $this;

    }

    /**
     * Проверка на совпадение и получение fileName
     *
     * @param string|null $name
     * @return string|null
     */
    public function checkFileName(?string $name): ?string
    {
        if (empty($name)) {
            return null;
        }

        if (!empty($this->id)) {
            connect()->where($this->primaryKey, '<>', $this->id);
        }

        if ($name = connect($this->table)->select('name')->where('formId', $this->formId)->where('LOWER(name)', strtolower($name))->getValue()) {
            return $name;
        }

        return null;
    }

    public function prepare(array $data, int $managerId): bool
    {
        // Установите детали модификации
        $currentTimestamp = gmdate('Y-m-d H:i:s');
        $data['modifiedBy']   = $managerId;
        $data['modifiedOn']   = $currentTimestamp;

        if (empty($this->id)) {
            // Задайте сведения о создании новых записей
            $data['addedBy']       = $managerId;
            $data['addedOn']       = $currentTimestamp;
            $data['orderBy']       = $this->getMaxValue('orderBy') + 1;

            $this->id = $this->insert($data);

            return true;
        } else {
            if ($this->update($data)) {
                return true;
            }
        }

        return false;
    }


    /**
     * Удаление
     *
     * @return bool
     * @throws NotFoundException
     * @throws NoContentException
     */
    public function remove(): bool
    {
        // Проверьте, доступен ли идентификатор поля
        if (empty($this->id)) {
            throw new NotFoundException("Not Found: ID is required.");
        }

        // Выборка поля по первичному ключу `fieldId`
        $currentRecord = connect($this->table)->where($this->primaryKey, $this->id)->get('array');

        // Throw exception if no record is found for the given `id`
        if (!$currentRecord) {
            throw new NoContentException("No Content: Record not found for the given ID.");
        }

        // Удалите поле и связанные с ним пользовательские значения
        connect()->table($this->table)->where($this->primaryKey, $this->id)->delete();

        return true;
    }



    /** STATIC METHODS */


    public static function getFieldTypes(): array
    {
        return self::$fieldTypes;
    }

    public static function getFieldTypeName($type): string
    {
        return self::$fieldTypes[$type] ?? '';
    }

    public static function getInputTypes(): array
    {
        return self::$inputTypes;
    }

    public static function getInputTypeName($type): string
    {
        return self::$inputTypes[$type] ?? '';
    }

    public static function getButtonTypes(): array
    {
        return self::$buttonTypes;
    }

    public static function getButtonTypeName($type): string
    {
        return self::$buttonTypes[$type] ?? '';
    }

    /**
     * @param int $typeId
     * @param string|null $status
     * @return array|null
     */
    public static function fields(int $typeId, ?string $status): ?array
    {
        if (!empty($status)) connect()->where('status', $status);

        return connect('form_fields')->where('formId', $typeId)->orderBy('orderBy ASC')->getAll('array');
    }


    /**
     * Получение значений
     *
     * @param string|null $options json
     * @return string Строка значений, разделенных запятой, либо пустая строка, если значений нет
     */
    public static function options(?string $options): string
    {
        $result = '';
        $displayLimit = 3; // лимит отображаемых размеров

        if (empty($options)) {
            return $result;
        }

        $options = json_decode($options, true);

        if (!empty($options)) {
            $options = collect($options)->pluck('label')->all();

            // Выбираем нужные размеры и формируем строку для вывода
            if (count($options) > $displayLimit) {
                // Выводим первые $displayLimit размеров и добавляем "и еще +X"
                $displayedSizes = collect($options)->splice(0, $displayLimit)->all();

                $remainingCount = count($options) - $displayLimit;
                $result = collect($displayedSizes)->implode(', ') . " и еще +{$remainingCount}";
            } else {
                // Если размеров меньше лимита, просто выводим их все
                $result = collect($options)->implode(', ');
            }
        }

        return $result;
    }


}