<?php
/**
 * Smarty plugin
 *
 * @package    Smarty
 * @subpackage PluginsModifier
 */
/**
 * Smarty date_format modifier plugin
 * Type:     modifier
 * Name:     date_format
 * Purpose:  format datestamps via strftime
 * Input:
 *          - string: input date string
 *          - format: strftime format for output
 *          - default_date: default date if $string is empty
 *
 * @link   https://www.smarty.net/manual/en/language.modifier.date.format.php date_format (Smarty online manual)
 * @author Monte Ohrt <monte at ohrt dot com>
 *
 * @param string $string       input date string
 * @param string $format       strftime format for output
 * @param string $default_date default date if $string is empty
 * @param string $formatter    either 'strftime' or 'auto'
 *
 * @return string |void
 * @uses   smarty_make_timestamp()
 */
function smarty_modifier_date_format($string, $format = null, $default_date = '', $formatter = 'auto')
{
    if ($format === null) {
        $format = Smarty::$_DATE_FORMAT;
    }

    /**
     * require_once the {@link shared.make_timestamp.php} plugin
     */
    static $is_loaded = false;
    if (!$is_loaded) {
        if (!is_callable('smarty_make_timestamp')) {
            include_once SMARTY_PLUGINS_DIR . 'shared.make_timestamp.php';
        }
        $is_loaded = true;
    }

    if ($string !== '' && $string !== '0000-00-00' && $string !== '0000-00-00 00:00:00') {
        $timestamp = smarty_make_timestamp($string);
    } elseif ($default_date !== '') {
        $timestamp = smarty_make_timestamp($default_date);
    } else {
        return;
    }

    // Если форматтер соответствует 'strftime', используем IntlDateFormatter или date()
    if ($formatter === 'strftime' || ($formatter === 'auto' && strpos($format, '%') !== false)) {
        $intlFormat = strftime_to_intl($format); // Преобразование strftime формата в Intl формат
        return (new \IntlDateFormatter(
            null, // Локаль используется текущая
            \IntlDateFormatter::NONE,
            \IntlDateFormatter::NONE,
            date_default_timezone_get(), // Текущая временная зона
            null,
            $intlFormat
        ))->format($timestamp);
    } else {
        // Используем стандартную функцию date()
        return date($format, $timestamp);
    }
}

/**
 * Преобразует strftime стиль в Intl-совместимый формат
 *
 * @param string $format Формат в стиле strftime
 * @return string Формат, совместимый с IntlDateFormatter
 */
function strftime_to_intl(string $format): string
{
    $replacements = [
        '%A' => 'EEEE',         // Полное название дня недели
        '%a' => 'EEE',          // Сокращенное название дня недели
        '%B' => 'LLLL',         // Полное название месяца
        '%b' => 'LLL',          // Сокращенное название месяца
        '%C' => 'yy',           // Первые две цифры года (века)
        '%d' => 'dd',           // День месяца (2 цифры, с ведущим нулем)
        '%e' => 'd',            // День месяца (с пробелом, без ведущего нуля)
        '%F' => 'yyyy-MM-dd',   // Краткий ISO 8601 формат даты
        '%H' => 'HH',           // Часы (24-часовой формат)
        '%I' => 'hh',           // Часы (12-часовой формат)
        '%j' => 'D',            // День года
        '%m' => 'MM',           // Месяц (с ведущим нулем)
        '%M' => 'mm',           // Минуты (с ведущим нулем)
        '%p' => 'a',            // AM/PM
        '%S' => 'ss',           // Секунды (с ведущим нулем)
        '%Y' => 'yyyy',         // Полный год
        '%y' => 'yy',           // Год (2 последние цифры)
        '%Z' => 'z',            // Часовой пояс
        '%z' => 'Z',            // Смещение часового пояса
    ];

    // Заменяем strftime формат на совместимый с Intl
    return strtr($format, $replacements);
}
//function smarty_modifier_date_format($string, $format = null, $default_date = '', $formatter = 'auto')
//{
//    if ($format === null) {
//        $format = Smarty::$_DATE_FORMAT;
//    }
//
//    // Load the shared.make_timestamp plugin if not already loaded
//    static $is_loaded = false;
//    if (!$is_loaded) {
//        if (!is_callable('smarty_make_timestamp')) {
//            include_once SMARTY_PLUGINS_DIR . 'shared.make_timestamp.php';
//        }
//        $is_loaded = true;
//    }
//
//    if ($string !== '' && $string !== '0000-00-00' && $string !== '0000-00-00 00:00:00') {
//        $timestamp = smarty_make_timestamp($string);
//    } elseif ($default_date !== '') {
//        $timestamp = smarty_make_timestamp($default_date);
//    } else {
//        return null;
//    }
//
//    // Use DateTime for formatting
//    $datetime = (new DateTime())->setTimestamp($timestamp);
//
//    if ($formatter === 'strftime' || ($formatter === 'auto' && strpos($format, '%') !== false)) {
//        // Handle strftime-style format
//        $format = str_replace(
//            ['%a', '%A', '%d', '%e', '%j', '%u', '%w', '%U', '%V', '%W', '%b', '%B', '%m', '%C', '%y', '%Y', '%H', '%I', '%l', '%M', '%p', '%P', '%S', '%z', '%Z', '%c', '%x', '%X', '%%'],
//            ['D', 'l', 'd', 'j', 'z', 'N', 'w', 'U', 'W', 'W', 'M', 'F', 'm', 'y', 'y', 'Y', 'H', 'h', 'g', 'i', 'A', 'a', 's', 'O', 'T', 'c', 'm/d/Y', 'H:i:s', '%'],
//            $format
//        );
//    }
//
//    return $datetime->format($format);
//}
